package transcoder

import (
	"path/filepath"
	"time"
)

const kbps = 1000

type TranscoderConfig struct {
	FileBase           string `json:"file_base"`
	SegmentMaxDuration int    `json:"segment_max_duration"`
	SegmentMinDuration int    `json:"segment_min_duration"`
	PlaylistWindowSize int    `json:"playlist_window_size"`
	PlaylistGeneration bool   `json:"playlist_generation"`
	EncoderType        string `json:"x264"`
	AssetContent       string `json:"asset_content"`
	AudioNormalize     bool   `json:"normalize_audio,omitempty"`
	FileExtension      string `json:"file_extension,omitempty"`

	AudioOnly  RenditionConfig   `json:"audio_only,omitempty"`
	Thumbnail  ThumbConfig       `json:"thumbnail,omitempty"`
	Transcodes []RenditionConfig `json:"transcodes"`
}

type RenditionConfig struct {
	FileBase           string `json:"file_base"`
	Label              string `json:"label"`
	DisplayName        string `json:"display_name,omitempty"`
	PlaylistPreference int    `json:"playlist_preference,omitempty"`
	Profile            string `json:"profile,omitempty"`
	Preset             string `json:"preset,omitempty"`
	MaxFPS             int    `json:"max_fps,omitempty"`
	Height             int    `json:"height,omitempty"`
	Bitrate            int    `json:"bitrate,omitempty"`
	OptimizeTS         bool   `json:"optimize_ts"`
	Validate           bool   `json:"validate"`
	FileNamePrefix     string `json:"file_name_prefix"`
	PlaylistGeneration bool   `json:"playlist_generation,omitempty"`
	Muxrate            int    `json:"muxrate,omitempty"`
	AudioChannels      int    `json:"audio_channels,omitempty"`
	AudioBitrate       int    `json:"audio_bitrate,omitempty"`
	Swsflags           string `json:"sws_flags,omitempty"`
	EncoderThreads     int    `json:"encoder_threads,omitempty"`
}

type ThumbConfig struct {
	Label              string  `json:"label"`
	FileBase           string  `json:"file_base"`
	FileNamePrefix     string  `json:"file_name_prefix"`
	Delay              int     `json:"delay"`
	Interval           int     `json:"interval"`
	LuminanceThreshold float64 `json:"luminance_threshold"`
}

var DefaultAudioOnlyConfig = RenditionConfig{
	Label:          "audio_only",
	FileNamePrefix: "index",
	OptimizeTS:     true,
	Validate:       true,
}

var DefaultThumbnailConfig = ThumbConfig{
	Label:          "thumb",
	FileNamePrefix: "index",
	Delay:          3000,
	Interval:       120000,
	// The luminance value for black thumbnails is around 0.068 ~ 0.069, therefore setting the value to 0.07 should prevent most of black thumbnails
	LuminanceThreshold: 0.07,
}

func BitrateFilter(maxBitrate int, renditions []RenditionConfig) []RenditionConfig {
	filtered := []RenditionConfig{}
	for _, r := range renditions {
		if r.Bitrate <= maxBitrate {
			filtered = append(filtered, r)
		}
	}
	return filtered
}

func HeightFilter(maxHeight int, renditions []RenditionConfig) []RenditionConfig {
	filtered := []RenditionConfig{}
	for _, r := range renditions {
		if r.Height <= maxHeight {
			filtered = append(filtered, r)
		}
	}
	return filtered
}

var r1080p60 = RenditionConfig{
	Label:              "1080p60",
	DisplayName:        "1080p60",
	Profile:            "main",
	Preset:             "medium",
	MaxFPS:             60,
	Height:             1080,
	Bitrate:            5500 * kbps,
	AudioChannels:      0,
	AudioBitrate:       0,
	Muxrate:            0,
	EncoderThreads:     12,
	OptimizeTS:         true,
	Validate:           true,
	FileNamePrefix:     "index",
	PlaylistGeneration: true,
}

var r720p60 = RenditionConfig{
	Label:              "720p60",
	DisplayName:        "720p60",
	Profile:            "main",
	Preset:             "medium",
	MaxFPS:             60,
	Height:             720,
	Bitrate:            3100 * kbps,
	AudioChannels:      0,
	AudioBitrate:       0,
	Muxrate:            0,
	EncoderThreads:     12,
	OptimizeTS:         true,
	Validate:           true,
	FileNamePrefix:     "index",
	PlaylistGeneration: true,
}

var r720p = RenditionConfig{
	Label:              "720p",
	DisplayName:        "720p",
	Profile:            "main",
	Preset:             "slow",
	MaxFPS:             30,
	Height:             720,
	Bitrate:            2100 * kbps,
	AudioChannels:      0,
	AudioBitrate:       0,
	Muxrate:            0,
	EncoderThreads:     8,
	OptimizeTS:         true,
	Validate:           true,
	FileNamePrefix:     "index",
	PlaylistGeneration: true,
}

var r480p = RenditionConfig{
	Label:              "480p",
	DisplayName:        "480p",
	Profile:            "main",
	Preset:             "slow",
	MaxFPS:             30,
	Height:             480,
	Bitrate:            1200 * kbps,
	AudioChannels:      0,
	AudioBitrate:       0,
	Muxrate:            0,
	EncoderThreads:     4,
	OptimizeTS:         true,
	Validate:           true,
	FileNamePrefix:     "index",
	PlaylistGeneration: true,
}

var r360p = RenditionConfig{
	Label:              "360p",
	DisplayName:        "360p",
	Profile:            "main",
	Preset:             "slow",
	Swsflags:           "BICUBIC",
	MaxFPS:             30,
	Height:             360,
	Bitrate:            0,
	AudioChannels:      2,
	AudioBitrate:       32 * kbps,
	Muxrate:            630 * kbps,
	EncoderThreads:     4,
	OptimizeTS:         true,
	Validate:           true,
	FileNamePrefix:     "index",
	PlaylistGeneration: true,
}

var r144p = RenditionConfig{
	Label:              "144p",
	DisplayName:        "144p",
	Profile:            "main",
	Preset:             "slow",
	Swsflags:           "BICUBIC",
	MaxFPS:             30,
	Height:             144,
	Bitrate:            0,
	AudioChannels:      2,
	AudioBitrate:       24 * kbps,
	Muxrate:            230 * kbps,
	EncoderThreads:     4,
	OptimizeTS:         true,
	Validate:           true,
	FileNamePrefix:     "index",
	PlaylistGeneration: true,
}

var DefaultTranscoderConfig = TranscoderConfig{
	SegmentMinDuration: int(4 * time.Second / time.Millisecond),
	SegmentMaxDuration: int(4 * time.Second / time.Millisecond),
	PlaylistWindowSize: 200,
	PlaylistGeneration: true,
	EncoderType:        "x264",
	FileExtension:      "",
	AssetContent:       "",
	AudioNormalize:     false,
	AudioOnly:          DefaultAudioOnlyConfig,
	Thumbnail:          DefaultThumbnailConfig,
	Transcodes:         []RenditionConfig{r1080p60, r720p60, r720p, r480p, r360p, r144p},
}

func ConfigWithDir(c TranscoderConfig, dir string) TranscoderConfig {
	c.FileBase = dir
	c.AudioOnly.FileBase = filepath.Join(dir, c.AudioOnly.Label)
	c.Thumbnail.FileBase = filepath.Join(dir, c.Thumbnail.Label)

	for i, r := range c.Transcodes {
		c.Transcodes[i].FileBase = filepath.Join(dir, r.Label)
	}

	return c
}
