package transcoder

import (
	"path"
	"sort"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
)

const outputManifestName = "index-dvr.m3u8"

func toManifest(id string, config TranscoderConfig, output *Output) (Video, error) {
	byLabel := outputByLabel(output)

	var renditions []Rendition
	video := Video{
		Id:         id,
		Renditions: renditions,
	}

	for label, output := range byLabel {
		r, err := toRendition(config, label, output)
		if err != nil {
			return video, err
		}
		video.Renditions = append(video.Renditions, r)
		video.Duration = r.Duration
	}

	for _, thumb := range output.Thumbnails {
		video.Thumbnails = append(video.Thumbnails, Thumbnail{
			Path:   path.Join(config.Thumbnail.Label, thumb.Path),
			Offset: float64(thumb.Timestamp),
		})
	}

	return video, nil
}

func toRendition(conf TranscoderConfig, label string, output Output) (Rendition, error) {
	r := Rendition{
		Label: label,
		Path:  label + "/",
	}

	totalSize := 0
	totalFrames := 0
	totalDuration := 0.0

	for _, seg := range output.Segments {
		s := Segment{
			Index:    int32(seg.SegmentNumber),
			Name:     seg.SegmentName,
			Duration: seg.Duration / 1000.0,
		}
		r.Segments = append(r.Segments, s)
		totalSize += int(seg.SegmentSize)
		totalFrames += int(seg.FrameCount)
		totalDuration += seg.Duration
	}

	totalDuration /= 1000.0
	if int32(totalDuration) > 0 {
		r.Duration = int32(totalDuration)
		r.Bitrate = 8 * int32(float64(totalSize)/totalDuration)
		r.MaxFPS = float64(totalFrames) / totalDuration
	}

	for _, codec := range output.Codecs {
		if codec.Height > 0 {
			r.Height = int32(codec.Height)
		}
		if codec.Width > 0 {
			r.Width = int32(codec.Width)
		}
		if codec.VideoCodec != "" {
			r.VideoCodec = codec.VideoCodec
		}
		if codec.AudioCodec != "" {
			r.AudioCodec = codec.AudioCodec
		}
	}

	if conf.PlaylistGeneration {
		r.Manifest = outputManifestName
	}

	return r, nil
}

// Split up the segments by label.
func outputByLabel(output *Output) map[string]Output {
	byLabel := make(map[string]Output)
	for _, seg := range output.Segments {
		out := byLabel[seg.Label]
		out.Segments = append(out.Segments, seg)
		byLabel[seg.Label] = out
	}
	for _, codec := range output.Codecs {
		out := byLabel[codec.Label]
		out.Codecs = append(out.Codecs, codec)
		byLabel[codec.Label] = out
	}
	// Make sure renditions are sorted by index
	for _, output := range byLabel {
		sort.Sort(sortedSegments(output.Segments))
	}

	return byLabel
}

type sortedSegments []avdata.Segment

func (s sortedSegments) Len() int           { return len(s) }
func (s sortedSegments) Less(i, j int) bool { return s[i].SegmentNumber < s[j].SegmentNumber }
func (s sortedSegments) Swap(i, j int)      { s[i], s[j] = s[j], s[i] }
