package main_test

import (
	"errors"

	"github.com/maxbrunsfeld/counterfeiter/fixtures"
	"github.com/maxbrunsfeld/counterfeiter/fixtures/fixturesfakes"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

var _ = Describe("A Fake generated by counterfeiter", func() {
	var fake *fixturesfakes.FakeSomething

	BeforeEach(func() {
		fake = new(fixturesfakes.FakeSomething)
	})

	It("implements the interface", func() {
		var interfaceVal fixtures.Something = fake
		Expect(interfaceVal).NotTo(BeNil())
	})

	It("can have its behavior configured using stub functions", func() {
		fake.DoThingsStub = func(arg1 string, arg2 uint64) (int, error) {
			Expect(arg1).To(Equal("stuff"))
			Expect(arg2).To(Equal(uint64(5)))
			return 3, errors.New("the-error")
		}

		num, err := fake.DoThings("stuff", 5)

		Expect(num).To(Equal(3))
		Expect(err).To(Equal(errors.New("the-error")))
	})

	It("can have its return values configured", func() {
		fake.DoThingsReturns(3, errors.New("the-error"))

		num, err := fake.DoThings("stuff", 5)
		Expect(num).To(Equal(3))
		Expect(err).To(Equal(errors.New("the-error")))
	})

	It("returns zero values when no return value or stub is provided", func() {
		fake.DoNothing()
		num, err := fake.DoThings("stuff", 5)

		Expect(num).To(Equal(0))
		Expect(err).To(BeNil())
	})

	It("allows overriding previous stub functions with return values", func() {
		fake.DoThingsStub = func(arg1 string, arg2 uint64) (int, error) {
			return 3, errors.New("the-error")
		}

		fake.DoThingsReturns(4, errors.New("other-error"))

		num, err := fake.DoThings("stuff", 5)
		Expect(num).To(Equal(4))
		Expect(err).To(Equal(errors.New("other-error")))
	})

	It("records the arguments it was called with", func() {
		Expect(fake.DoThingsCallCount()).To(Equal(0))

		fake.DoThings("stuff", 5)

		Expect(fake.DoThingsCallCount()).To(Equal(1))
		arg1, arg2 := fake.DoThingsArgsForCall(0)
		Expect(arg1).To(Equal("stuff"))
		Expect(arg2).To(Equal(uint64(5)))
	})

	It("records a slice argument as a copy", func() {
		buffer := []byte{1}

		fake.DoASlice(buffer)

		buffer[0] = 2
		arg1 := fake.DoASliceArgsForCall(0)
		Expect(arg1).To(ConsistOf(byte(1)))
	})

	It("records a nil slice argument as a nil", func() {
		var buffer []byte = nil

		fake.DoASlice(buffer)

		arg1 := fake.DoASliceArgsForCall(0)
		Expect(arg1).To(BeNil())
	})

	It("records an array argument as a copy", func() {
		buffer := [4]byte{1, 2, 3, 4}

		fake.DoAnArray(buffer)

		buffer[0] = 2
		arg1 := fake.DoAnArrayArgsForCall(0)
		Expect(arg1).To(ConsistOf(byte(1), byte(2), byte(3), byte(4)))
	})

	It("passes the original slice to a stub function", func() {
		buffer := []byte{1}

		fake.DoASliceStub = func(b []byte) {
			b[0] = 2
		}

		fake.DoASlice(buffer)

		arg1 := fake.DoASliceArgsForCall(0)
		Expect(arg1).To(ConsistOf(byte(1)))
		Expect(buffer).To(ConsistOf(byte(2)))
	})

	It("records its calls without race conditions", func() {
		go fake.DoNothing()

		Eventually(fake.DoNothingCallCount, 1.0).Should(Equal(1))
	})

	Describe("implementing an interface to show recorded methoded invocations", func() {
		var (
			invocationRecorder InvocationRecorder
			ok                 bool
		)

		BeforeEach(func() {
			var ifake interface{} = fake
			invocationRecorder, ok = ifake.(InvocationRecorder)

			Expect(ok).To(BeTrue())
		})

		It("records each invocation", func() {
			Expect(len(fake.Invocations()["DoThings"])).To(Equal(0))
			Expect(len(fake.Invocations()["DoNothing"])).To(Equal(0))
			Expect(len(fake.Invocations()["DoASlice"])).To(Equal(0))
			Expect(len(fake.Invocations()["DoAnArray"])).To(Equal(0))

			fake.DoThings("hello", 0)
			Expect(len(fake.Invocations()["DoThings"])).To(Equal(1))
			Expect(fake.Invocations()["DoThings"][0][0]).To(Equal("hello"))
			Expect(fake.Invocations()["DoThings"][0][1]).To(Equal(uint64(0)))

			fake.DoNothing()
			Expect(len(fake.Invocations()["DoNothing"])).To(Equal(1))

			fake.DoASlice([]byte("HAI"))
			Expect(len(fake.Invocations()["DoASlice"])).To(Equal(1))
			Expect(fake.Invocations()["DoASlice"][0][0]).To(Equal([]byte("HAI")))

			fake.DoAnArray([4]byte{})
			Expect(len(fake.Invocations()["DoAnArray"])).To(Equal(1))
			Expect(fake.Invocations()["DoAnArray"][0][0]).ToNot(BeNil())
		})
	})

	Describe("when two methods are called at the same time", func() {
		It("does not deadlock", func(done Done) {
			start1 := make(chan struct{})
			end1 := make(chan struct{})
			start2 := make(chan struct{})
			end2 := make(chan struct{})

			fake.DoNothingStub = func() {
				close(start1)
				<-end1
			}

			fake.DoThingsStub = func(string, uint64) (int, error) {
				close(start2)
				<-end2
				return 0, nil
			}

			go fake.DoNothing()
			<-start1
			go fake.DoThings("", 1)
			<-start2

			close(end2)
			close(end1)
			close(done)
		})

		Describe("and the invocations of those methods are recorded", func() {
			BeforeEach(func() {
				go func() {
					fake.DoNothing()
					fake.DoThings("abc", 1)
					fake.DoASlice([]byte{})
				}()
			})

			It("records the call count without race conditions", func() {
				Eventually(fake.DoNothingCallCount).Should(Equal(1))
				Eventually(fake.DoThingsCallCount).Should(Equal(1))
				Eventually(fake.DoASliceCallCount).Should(Equal(1))
			})

			It("records the invocations without race conditions as well", func() {
				Eventually(func() [][]interface{} { return fake.Invocations()["DoNothing"] }).Should(HaveLen(1))
				Eventually(func() [][]interface{} { return fake.Invocations()["DoThings"] }).Should(HaveLen(1))
				Eventually(func() [][]interface{} { return fake.Invocations()["DoASlice"] }).Should(HaveLen(1))
			})
		})
	})

	Describe("when the same method is called concurrently", func() {
		It("does not deadlock", func(done Done) {
			defer close(done)

			a := make(chan struct{})
			b := make(chan struct{})

			fake.DoNothingStub = func() {
				select {
				case <-a:
					close(b)
				default:
					close(a)
					<-b
				}
			}

			go fake.DoNothing()
			go fake.DoNothing()

			<-b
		})
	})

	Describe("interfaces with var-args methods", func() {
		var fake *fixturesfakes.FakeHasVarArgs

		BeforeEach(func() {
			fake = new(fixturesfakes.FakeHasVarArgs)
		})

		It("implements the interface", func() {
			var interfaceVal fixtures.HasVarArgs = fake
			Expect(interfaceVal).NotTo(BeNil())
		})

		It("records the calls in a slice", func() {
			fake.DoThings(5, "one", "two", "three")

			num, strings := fake.DoThingsArgsForCall(0)
			Expect(num).To(Equal(5))
			Expect(strings).To(Equal([]string{"one", "two", "three"}))
		})

		It("passes the var-args to stub functions", func() {
			fake.DoThingsStub = func(x int, strings ...string) int {
				Expect(strings).To(Equal([]string{"one", "two", "three"}))
				return 11
			}

			val := fake.DoThings(5, "one", "two", "three")
			Expect(val).To(Equal(11))
		})
	})
})

type InvocationRecorder interface {
	Invocations() map[string][][]interface{}
}
