package integration_test

import (
	"io"
	"io/ioutil"
	"os"
	"os/exec"
	"path/filepath"

	"github.com/onsi/gomega/gbytes"
	"github.com/onsi/gomega/gexec"

	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

var _ = Describe("The counterfeiter CLI", func() {
	var pathToCLI string

	BeforeEach(func() {
		pathToCLI = tmpPath("counterfeiter")
	})

	It("can generate a fake for a typed function", func() {
		copyIn("typed_function.go", pathToCLI)

		session := startCounterfeiter(pathToCLI, "typed_function.go", "SomethingFactory")

		Eventually(session).Should(gexec.Exit(0))
		Expect(session).To(gbytes.Say("Wrote `FakeSomethingFactory"))

		generatedFakePath := filepath.Join(pathToCLI, "fixtures", "fixturesfakes", "fake_something_factory.go")
		Expect(generatedFakePath).To(BeARegularFile())

		expectedOutputPath := "../fixtures/expected_output/fake_something_factory.example"
		expectedContents, err := ioutil.ReadFile(expectedOutputPath)
		Expect(err).ToNot(HaveOccurred())

		actualContents, err := ioutil.ReadFile(generatedFakePath)
		Expect(err).ToNot(HaveOccurred())

		// assert file content matches what we expect
		Expect(string(actualContents)).To(Equal(string(expectedContents)))
	})

	Describe("when given a single argument", func() {
		BeforeEach(func() {
			copyIn("other_types.go", pathToCLI)
			copyIn("something.go", tmpPath("otherrepo.com"))
		})

		It("writes a fake for the fully qualified interface that is provided in the argument", func() {
			session := startCounterfeiterWithoutFixture(pathToCLI, "otherrepo.com/fixtures.Something")

			Eventually(session).Should(gexec.Exit(0))
			output := string(session.Out.Contents())

			Expect(output).To(ContainSubstring("Wrote `FakeSomething`"))
		})
	})

	Describe("when given two arguments", func() {
		BeforeEach(func() {
			copyIn("something.go", pathToCLI)
		})

		It("writes a fake for the given interface from the provided file", func() {
			session := startCounterfeiter(pathToCLI, "something.go", "Something")

			Eventually(session).Should(gexec.Exit(0))
			output := string(session.Out.Contents())

			Expect(output).To(ContainSubstring("Wrote `FakeSomething`"))
		})
	})

	Describe("when provided three arguments", func() {
		BeforeEach(func() {
			copyIn("something.go", pathToCLI)
		})

		It("writes the fake to stdout", func() {
			session := startCounterfeiter(pathToCLI, "something.go", "Something", "-")

			Eventually(session).Should(gexec.Exit(0))
			output := string(session.Out.Contents())

			Expect(output).To(ContainSubstring("// Code generated by counterfeiter. DO NOT EDIT."))
		})
	})
})

// helper functions

func tmpPath(destination string) string {
	return filepath.Join(tmpDir, "src", destination)
}

func copyIn(fixture string, destination string) {
	destination = filepath.Join(destination, "fixtures")
	err := os.MkdirAll(destination, 0777)
	Expect(err).ToNot(HaveOccurred())
	filepath.Walk(filepath.Join("..", "fixtures", fixture), func(path string, info os.FileInfo, err error) error {
		if info.IsDir() {
			return nil
		}

		base := filepath.Base(path)

		src, err := os.Open(path)
		Expect(err).ToNot(HaveOccurred())

		dst, err := os.Create(filepath.Join(destination, base))
		Expect(err).ToNot(HaveOccurred())

		_, err = io.Copy(dst, src)
		Expect(err).ToNot(HaveOccurred())
		return nil
	})
}

func startCounterfeiter(workingDir string, fixtureName string, otherArgs ...string) *gexec.Session {
	fakeGoPathDir := filepath.Dir(filepath.Dir(workingDir))
	absPath, _ := filepath.Abs(fakeGoPathDir)
	absPathWithSymlinks, _ := filepath.EvalSymlinks(absPath)

	fixturePath := filepath.Join("fixtures", fixtureName)
	args := append([]string{fixturePath}, otherArgs...)
	cmd := exec.Command(pathToCounterfeiter, args...)
	cmd.Dir = workingDir
	cmd.Env = []string{"GOPATH=" + absPathWithSymlinks}

	session, err := gexec.Start(cmd, GinkgoWriter, GinkgoWriter)
	Expect(err).ToNot(HaveOccurred())
	return session
}

func startCounterfeiterWithoutFixture(workingDir string, arg string) *gexec.Session {
	fakeGoPathDir := filepath.Dir(filepath.Dir(workingDir))
	absPath, _ := filepath.Abs(fakeGoPathDir)
	absPathWithSymlinks, _ := filepath.EvalSymlinks(absPath)

	cmd := exec.Command(pathToCounterfeiter, arg)
	cmd.Dir = workingDir
	cmd.Env = []string{"GOPATH=" + absPathWithSymlinks}

	session, err := gexec.Start(cmd, GinkgoWriter, GinkgoWriter)
	Expect(err).ToNot(HaveOccurred())
	return session
}

// gexec setup

var tmpDir string
var pathToCounterfeiter string

var _ = SynchronizedBeforeSuite(func() []byte {
	pathToCounterfeiter, err := gexec.Build("github.com/maxbrunsfeld/counterfeiter")
	Expect(err).ToNot(HaveOccurred())
	return []byte(pathToCounterfeiter)
}, func(computedPath []byte) {
	pathToCounterfeiter = string(computedPath)
})

var _ = SynchronizedAfterSuite(func() {}, func() {
	gexec.CleanupBuildArtifacts()
})

var _ = BeforeEach(func() {
	var err error
	tmpDir, err = ioutil.TempDir("", "counterfeiter-integration")
	Expect(err).ToNot(HaveOccurred())
})

var _ = AfterEach(func() {
	err := os.RemoveAll(tmpDir)
	Expect(err).ToNot(HaveOccurred())
})
