# ECS Module
#
# Uses the "ecsasg" module in to set up the EC2 auto-scaling group for ECS.
# The actual ECS cluster is created here.

terraform {
  required_version = ">= 0.12"
}


data "aws_iam_account_alias" "current" {}

data "aws_caller_identity" "current" {}

### Logging

resource "aws_iam_role_policy_attachment" "allow_send_access_logs_policy" {
  role       = var.ecs_host_iam_id
  policy_arn = "arn:aws:iam::aws:policy/AWSOpsWorksCloudWatchLogs"
}

resource "aws_cloudwatch_log_group" "ecs_agent_logs" {
  name              = "${var.cluster_name}-ecs-agent-logs"
  retention_in_days = var.log_retention_days

  tags = {
    app     = var.app_name
    cluster = var.cluster_name
    env     = var.environment
  }
}

resource "aws_cloudwatch_log_group" "ecs_init_logs" {
  name              = "${var.cluster_name}-ecs-init-logs"
  retention_in_days = var.log_retention_days

  tags = {
    app     = var.app_name
    cluster = var.cluster_name
    env     = var.environment
  }
}

resource "aws_cloudwatch_log_group" "ssm_logs" {
  name              = "${var.cluster_name}-ssm-ec2-logs"
  retention_in_days = var.log_retention_days

  tags = {
    app     = var.app_name
    cluster = var.cluster_name
    env     = var.environment
  }
}

### Cluster ASG

# Set the internal LDAP host for EC2 ssh auth based on the current account
locals {
  acct             = data.aws_iam_account_alias.current.account_alias
  custom_ldap_host = var.ldap_hosts[local.acct]
}

# Define how our upcoming ECS cluster will get autoscaled & how the cluster's EC2's are deployed and configured
# Note: installs LDAP-AUTH on the ec2's pointing to our special LDAP hosts
module "ecs_asg" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/edge/terraform//ecsasg?ref=v3.1.0"

  cluster_name                  = var.cluster_name
  task_cleanup_wait_duration    = var.task_cleanup_wait_duration
  task_shutdown_wait_duration   = var.task_shutdown_wait_duration
  container_instance_profile_id = var.ecs_host_iam_id
  desired_capacity              = var.cluster_size["desired"]
  max_size                      = var.cluster_size["max"]
  min_size                      = var.cluster_size["min"]
  ami                           = var.ami
  environment                   = var.environment
  instance_type                 = var.ec2_instance_type
  ldap_group                    = var.ssh_sudoer_ldap_group
  custom_ldap_host              = local.custom_ldap_host
  name                          = var.cluster_name
  owner                         = var.owner_email

  ecs_agent_loggroup = aws_cloudwatch_log_group.ecs_agent_logs.name
  ecs_init_loggroup  = aws_cloudwatch_log_group.ecs_agent_logs.name
  ssm_agent_loggroup = aws_cloudwatch_log_group.ssm_logs.name
  team               = var.team_name

  security_groups = join(",", var.security_groups)

  # Subnets to which ASG will add new ec2's. Yeah the var name is weird
  vpc_zone_identifier = join(",", var.subnets)

  # How do we scale up in case of high cpu?
  scale_out_cooldown           = 1200
  scale_out_evaluation_periods = 5
  scale_out_increment          = 1

  # set really high - let's avoid unnecessary scaling up
  scale_out_cpu_percent_threshold = 90

  # How do we scale down in case of low cpu?
  scale_in_cooldown           = 900
  scale_in_evaluation_periods = 15
  scale_in_increment          = -1

  # set really low - let's avoid unnecessary scaling down
  scale_in_cpu_percent_threshold = 0.01
}

### ECS Cluster

resource "aws_ecs_cluster" "main" {
  name = var.cluster_name
}

### ECS Service

resource "aws_iam_role" "ecs_task_execution_role" {
  name = "ecs_task_execution_${var.app_name}_${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "ecs_task_execution_role" {
  role       = aws_iam_role.ecs_task_execution_role.id
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_ecs_task_definition" "app" {
  family                = var.cluster_name
  network_mode          = var.network_mode
  execution_role_arn    = aws_iam_role.ecs_task_execution_role.arn
  task_role_arn         = var.ecs_task_role
  container_definitions = var.container_definitions

  lifecycle {
    ignore_changes = [container_definitions]
  }
}

resource "aws_ecs_service" "main" {
  name            = var.cluster_name
  cluster         = aws_ecs_cluster.main.id
  task_definition = aws_ecs_task_definition.app.arn
  desired_count   = var.app_count
  iam_role        = var.service_iam_role

  lifecycle {
    ignore_changes = [task_definition]
  }

  ordered_placement_strategy {
    type  = "spread"
    field = "instanceId"
  }

  load_balancer {
    target_group_arn = var.target_group_id
    container_name   = var.cluster_name
    container_port   = var.app_port
  }
}
