terraform {
  required_version = ">= 0.12"
}

resource "aws_alb" "main" {
  name            = var.cluster_name
  internal        = false
  subnets         = var.subnets
  security_groups = var.security_groups

  enable_deletion_protection = true
}

resource "aws_alb_target_group" "main" {
  deregistration_delay = 5 # seconds
  slow_start           = var.slow_start_seconds
  name                 = var.cluster_name
  protocol             = "HTTP"
  vpc_id               = var.vpc_id

  # The port is required, but overriden by ECS. Using a placeholder here.
  port = "1"

  health_check {
    protocol            = "HTTP"
    path                = "/health"
    healthy_threshold   = 3   # checks resulting in success
    unhealthy_threshold = 2   # checks resulting in failure
    timeout             = 2   # seconds
    interval            = 5   # seconds
    matcher             = 200 # status code
  }
}

resource "aws_alb_listener" "https" {
  load_balancer_arn = aws_alb.main.id
  port              = "443"
  protocol          = "HTTPS"
  certificate_arn   = aws_acm_certificate.ssl_cert.arn

  default_action {
    type = "fixed-response"

    fixed_response {
      status_code  = 404
      message_body = "Not Found"
      content_type = "text/plain"
    }
  }
}

resource "aws_alb_listener_rule" "https_listen_only" {
  listener_arn = aws_alb_listener.https.arn

  action {
    type             = "forward"
    target_group_arn = aws_alb_target_group.main.id
  }

  condition {
    field  = "path-pattern"
    values = ["/v1/connect"]
  }
}


# Request an SSL certificate through ACM.
# To validate a certificate, create the CNAME DNS record required by the DNS validation method
# in the internal DNS dashboard. See https://wiki.twitch.com/x/CoFQB.
resource "aws_acm_certificate" "ssl_cert" {
  domain_name       = var.domain_name
  validation_method = "DNS"

  tags = {
    app         = var.app_name
    environment = var.environment
  }

  lifecycle {
    create_before_destroy = true
  }
}

