package main

import (
	"time"

	"code.justin.tv/hygienic/distconf"
)

type config struct {
	awsRegion            string
	maxUsers             int
	suiteTimeout         time.Duration
	numMessages          int
	messageInterval      time.Duration
	userInterval         time.Duration
	subscriptionInterval time.Duration
	autoprofBucket       string
}

// Configuration will be read from top to bottom of the readers list.
func loadConfig(log distconf.Logger) *config {
	readers := []distconf.Reader{&distconf.Env{}}
	d := &distconf.Distconf{Logger: log, Readers: readers}

	maxUsers := int(d.Int("MAX_USERS", 1).Get())
	userInterval := time.Duration(d.Int("USER_INTERVAL_MILLISECONDS", 5).Get()) * time.Millisecond
	numMessages := int(d.Int("NUM_MESSAGES", 100).Get())
	messageInterval := time.Duration(d.Int("MESSAGE_INTERVAL_SECONDS", 20).Get()) * time.Second
	return &config{
		awsRegion:      d.Str("AWS_REGION", "us-west-2").Get(),
		autoprofBucket: d.Str("AUTOPROF_BUCKET", "").Get(),

		maxUsers:             maxUsers,
		numMessages:          numMessages,
		messageInterval:      messageInterval,
		userInterval:         userInterval,
		subscriptionInterval: time.Duration(d.Int("SUBSCRIPTION_INTERVAL_SECONDS", 5).Get()) * time.Second,

		// Suite timeout = time until last user starts + duration for a user + period of leeway
		suiteTimeout: time.Duration(maxUsers)*userInterval + time.Duration(numMessages)*messageInterval + 10*time.Minute,
	}
}
