# Entrypoint for the websocket edge production infrastructure
# Only supports us-west-2 region

module "constants" {
  source = "../../modules/constants"
}

locals {
  region                 = "us-west-2"
  environment            = "prod"
  account_alias          = "twitch-websocket-edge-prod"
  account_id             = 636874578776
  cluster_name           = "${module.constants.app_name}-${local.environment}"
  load_test_cluster_name = "${module.constants.load_test_name}-${local.environment}"
  disconnect_queue_name  = "disconnect-queue"

  domain_name                 = "ws.internal.justin.tv"
  task_shutdown_wait_duration = "5m"
}

provider "aws" {
  version             = "~> 2.23.0"
  region              = "us-west-2"
  allowed_account_ids = [636874578776]
}

terraform {
  required_version = ">= 0.12"

  backend "s3" {
    bucket = "twitch-websocket-edge-prod"
    key    = "tfstate/edge/twitch-websocket-edge-prod/prod/terraform.tfstate"
    region = "us-west-2"
  }
}

locals {
  # used to prime the production env - will be ignored in favor of latest deploy in the future
  image_tag     = "67f2e3b" # Ignored. See module/ecs/main.tf's aws_ecs_service.main.lifecycle.ignore_changes
  app_count     = 3
  instance_type = "c5n.9xlarge"
  task_cpu      = 36864
  task_memory   = 93533 # Not using the full 94533 MiB, to save space for the ECS agent.
}

// Cluster size must be at least one greater than app_count, to allow for rolling deploys to occur.
variable "cluster_size" {
  default = {
    min     = 4
    max     = 4
    desired = 4
  }
}

data "terraform_remote_state" "base_account" {
  backend = "s3"

  config = {
    bucket = "twitch-websocket-edge-prod"
    key    = "tfstate/account_template/prod-us-west-2/terraform.tfstate"
    region = "us-west-2"
  }
}

resource "aws_security_group" "public_internet" {
  name        = "public_internet_https"
  description = "Allow all public internet HTTPS traffic."
  vpc_id      = data.terraform_remote_state.base_account.outputs.us_west_2_vpc_id

  ingress {
    from_port = 443
    to_port   = 443
    protocol  = "tcp"

    # allow all traffic in on the HTTPS port
    cidr_blocks      = ["0.0.0.0/0"]
    ipv6_cidr_blocks = ["::/0"]
  }

  egress {
    from_port = 0
    to_port   = 65535
    protocol  = "tcp"

    # Allow all traffic out
    cidr_blocks      = ["0.0.0.0/0"]
    ipv6_cidr_blocks = ["::/0"]
  }
}

module "external_alb" {
  source = "../../modules/external_alb"

  cluster_name    = local.cluster_name
  domain_name     = local.domain_name
  subnets         = data.terraform_remote_state.base_account.outputs.us_west_2_public_subnets
  security_groups = [aws_security_group.public_internet.id]
  vpc_id          = data.terraform_remote_state.base_account.outputs.us_west_2_vpc_id
  app_name        = module.constants.app_name
  environment     = local.environment
  logs_bucket     = "twitch-websocket-edge-prod-logs" # TODO: unconstantify
}

// TODO: remove dependency, this doesn't bring in that much utility.
# Set up all the roles and permissions for a standard ECS cluster
# NOTE: if we expand to multiple regions, only instantiate this in one region.
# This module only contains global resources like IAM, so we do not need one per region.
module "core_account" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/edge/terraform//core_account?ref=v3.0.1"

  environment = local.environment
  team        = module.constants.team_name
  xray_role   = "*"
}

module "websocket_edge" {
  source = "../../modules/websocket-edge"

  app_name     = module.constants.app_name
  cluster_name = local.cluster_name
  environment  = local.environment
  app_image    = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/websocket-edge:${local.image_tag}"
  task_memory  = local.task_memory
  task_cpu     = local.task_cpu

  # The ALB terminates TLS, so our app listens on 80.
  serving_port         = 80
  disconnect_queue_url = module.constants.prod_gql_subs_disconnect_queue_url
  disconnect_queue_arn = module.constants.prod_gql_subs_disconnect_queue_arn
  autoprof_bucket      = module.autoprof.s3_bucket
  gql_subs_url         = module.constants.gql_subs_url_prod
  target_group_id = module.external_alb.target_group_id
  load_balancer_id = module.external_alb.load_balancer_id
}

module "ecs" {
  source = "../../modules/ecs"

  app_name                    = module.constants.app_name
  cluster_name                = local.cluster_name
  network_mode                = "host"
  environment                 = local.environment
  app_count                   = local.app_count
  ec2_instance_type           = local.instance_type
  cluster_size                = var.cluster_size
  ecs_host_iam_id             = module.core_account.container_instance_profile_id
  target_group_id             = module.external_alb.target_group_id
  app_port                    = module.websocket_edge.app_port
  container_definitions       = module.websocket_edge.container_definitions
  ecs_task_role               = module.websocket_edge.ecs_task_role
  ldap_hosts                  = module.constants.ldap_hosts
  ssh_sudoer_ldap_group       = module.constants.ssh_sudoer_ldap_group
  owner_email                 = module.constants.owner_email
  service_iam_role            = module.core_account.service_iam_role
  task_shutdown_wait_duration = local.task_shutdown_wait_duration
  security_groups             = [data.terraform_remote_state.base_account.outputs.us_west_2_twitch_subnets_sg_id]
  subnets                     = data.terraform_remote_state.base_account.outputs.us_west_2_private_subnets
}

resource "aws_sqs_queue" "disconnect_queue_deadletter" {
  name                      = "${local.disconnect_queue_name}-deadletter"
  max_message_size          = 1024
  message_retention_seconds = 604800 # 7 days
}

resource "aws_sqs_queue" "disconnect_queue" {
  name                      = local.disconnect_queue_name
  max_message_size          = 1024
  message_retention_seconds = 86400
  redrive_policy            = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.disconnect_queue_deadletter.arn}\",\"maxReceiveCount\":4}"

  policy = <<EOF
{
   "Version": "2012-10-17",
   "Statement": [{
      "Effect": "Allow",
      "Principal": {
         "AWS": [
            "${module.constants.dev_gqlsubs_task_role}",
            "${module.constants.dev_gqlsubs_account_role}",
            "${module.constants.prod_gqlsubs_task_role}",
            "${module.constants.prod_gqlsubs_account_role}"
         ]
      },
      "Action": [
         "sqs:DeleteMessage",
         "sqs:ReceiveMessage",
         "sqs:ChangeMessageVisibility"
      ],
      "Resource": "arn:aws:sqs:${local.region}:${local.account_id}:${local.disconnect_queue_name}"
   }]
}
EOF


  tags = {
    Environment = local.environment
  }
}

// TODO: lifecycle policy on bucket.
module "autoprof" {
  source = "../../modules/autoprof"

  app_name   = module.constants.app_name
  account_id = local.account_id
  region     = local.region
}
