#include <alice/beggins/internal/bert/embedder.h>

#include <quality/relev_tools/bert_models/lib/compress_decompress_node.h>
#include <quality/relev_tools/bert_models/lib/one_file_package.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NAlice::NBeggins::NInternal;

class BertEmbedderTest : public ::testing::Test {
protected:
    void SetUp() override {
        Embedder = LoadEmbedder(TBlob::FromFileContent("test_models/model.bert.htxt"),
                                {.MaxBatchSize = 64, .ExpectedMaxSubBatchesNum = 1, .NumThreads = 4});
    }

    std::unique_ptr<TBertEmbedder> Embedder;
};

namespace {

constexpr size_t EMBEDDING_SIZE = 1024;

TString MakeBigQuery(size_t nWords) {
    nWords *= 2;
    auto query = TString(nWords, 'a');
    for (size_t i = 0; i < nWords; ++i) {
        if (i % 2 == 1) {
            query[i] = ' ';
        }
    }
    return query;
}

} // namespace

TEST_F(BertEmbedderTest, Process) {
    static const TVector<float> expectedEmbeddings = {
        0.207136869430542,    -0.19905543327331543, 1.8319058418273926,   0.6133291721343994,   0.207136869430542,
        -1.0114398002624512,  1.4257135391235352,   -2.2300164699554443,  -0.19905543327331543, -0.19905543327331543,
        1.0195212364196777,   -0.6052474975585938,  -1.0114398002624512,  -1.0114398002624512,  0.207136869430542,
        0.6133291721343994,   -0.19905543327331543, -2.6362085342407227,  -1.417631983757019,   -1.0114398002624512,
        -0.19905543327331543, 0.6133291721343994,   0.6133291721343994,   -1.0114398002624512,  1.0195212364196777,
        -0.19905543327331543, -0.19905543327331543, -0.6052474975585938,  -0.6052474975585938,  0.207136869430542,
        1.4257135391235352,   -1.0114398002624512,  -2.2300164699554443,  -1.823824167251587,   -1.0114398002624512,
        -0.6052474975585938,  1.0195212364196777,   -1.0114398002624512,  -1.417631983757019,   -1.0114398002624512,
        -1.823824167251587,   0.207136869430542,    -1.417631983757019,   -1.417631983757019,   -0.19905543327331543,
        1.8319058418273926,   0.207136869430542,    0.6133291721343994,   0.207136869430542,    -0.19905543327331543,
        -0.19905543327331543, 1.0195212364196777,   0.207136869430542,    0.6133291721343994,   1.0195212364196777,
        -0.19905543327331543, 0.207136869430542,    -0.6052474975585938,  1.8319058418273926,   0.6133291721343994,
        0.207136869430542,    -1.0114398002624512,  -0.6052474975585938,  -0.19905543327331543, 1.4257135391235352,
        -1.417631983757019,   1.8319058418273926,   1.0195212364196777,   -0.19905543327331543, -1.823824167251587,
        -1.417631983757019,   -0.6052474975585938,  -0.19905543327331543, 0.207136869430542,    -1.417631983757019,
        -1.417631983757019,   -1.0114398002624512,  -1.417631983757019,   1.4257135391235352,   1.0195212364196777,
        -0.19905543327331543, 0.207136869430542,    -0.19905543327331543, 0.207136869430542,    -0.6052474975585938,
        1.8319058418273926,   0.6133291721343994,   -1.0114398002624512,  -0.6052474975585938,  -0.6052474975585938,
        -1.0114398002624512,  0.207136869430542,    -0.19905543327331543, -0.6052474975585938,  1.4257135391235352,
        -2.2300164699554443,  -0.19905543327331543, 0.6133291721343994,   -1.0114398002624512,  0.6133291721343994,
        1.8319058418273926,   1.0195212364196777,   -1.823824167251587,   0.207136869430542,    -1.417631983757019,
        0.207136869430542,    -0.6052474975585938,  -0.19905543327331543, -1.417631983757019,   -0.19905543327331543,
        -0.19905543327331543, -0.19905543327331543, 1.0195212364196777,   -0.6052474975585938,  -0.19905543327331543,
        -0.19905543327331543, -0.19905543327331543, -1.417631983757019,   0.6133291721343994,   0.207136869430542,
        0.6133291721343994,   -2.2300164699554443,  0.207136869430542,    1.4257135391235352,   -0.6052474975585938,
        -1.823824167251587,   0.207136869430542,    -0.19905543327331543, 1.0195212364196777,   -1.0114398002624512,
        1.4257135391235352,   0.6133291721343994,   1.8319058418273926,   -1.417631983757019,   -0.19905543327331543,
        -1.417631983757019,   0.207136869430542,    -0.19905543327331543, -1.0114398002624512,  -1.0114398002624512,
        -1.0114398002624512,  -1.417631983757019,   0.6133291721343994,   0.207136869430542,    -0.19905543327331543,
        -1.417631983757019,   -1.417631983757019,   -0.6052474975585938,  -1.0114398002624512,  1.0195212364196777,
        1.0195212364196777,   -1.823824167251587,   0.207136869430542,    -0.19905543327331543, 0.207136869430542,
        -1.417631983757019,   0.6133291721343994,   -0.6052474975585938,  1.4257135391235352,   -1.823824167251587,
        -0.6052474975585938,  -1.417631983757019,   -1.0114398002624512,  0.6133291721343994,   2.23809814453125,
        -0.6052474975585938,  -1.823824167251587,   -0.19905543327331543, -1.417631983757019,   1.4257135391235352,
        -1.0114398002624512,  -1.0114398002624512,  -0.6052474975585938,  0.6133291721343994,   -1.0114398002624512,
        0.6133291721343994,   -0.19905543327331543, -1.0114398002624512,  -0.6052474975585938,  -0.19905543327331543,
        0.207136869430542,    -0.6052474975585938,  1.0195212364196777,   0.6133291721343994,   0.6133291721343994,
        -1.417631983757019,   1.4257135391235352,   1.0195212364196777,   1.0195212364196777,   -0.6052474975585938,
        0.6133291721343994,   -0.19905543327331543, 1.8319058418273926,   -1.823824167251587,   0.6133291721343994,
        0.207136869430542,    0.207136869430542,    -1.0114398002624512,  -0.19905543327331543, -1.0114398002624512,
        0.207136869430542,    -0.6052474975585938,  -1.417631983757019,   -1.0114398002624512,  -1.823824167251587,
        -2.2300164699554443,  0.207136869430542,    0.6133291721343994,   0.6133291721343994,   -1.0114398002624512,
        -1.0114398002624512,  0.207136869430542,    -0.6052474975585938,  0.207136869430542,    0.6133291721343994,
        -0.6052474975585938,  -0.19905543327331543, -0.6052474975585938,  -0.19905543327331543, -0.6052474975585938,
        1.4257135391235352,   -0.6052474975585938,  2.23809814453125,     -1.417631983757019,   -0.6052474975585938,
        0.207136869430542,    -2.6362085342407227,  0.6133291721343994,   0.207136869430542,    -0.19905543327331543,
        -0.19905543327331543, 0.207136869430542,    -2.2300164699554443,  0.6133291721343994,   -0.6052474975585938,
        -1.0114398002624512,  -0.6052474975585938,  0.207136869430542,    -1.823824167251587,   -0.19905543327331543,
        0.207136869430542,    -0.19905543327331543, 0.207136869430542,    0.207136869430542,    -0.19905543327331543,
        -1.823824167251587,   2.23809814453125,     0.6133291721343994,   1.0195212364196777,   -0.6052474975585938,
        0.6133291721343994,   1.8319058418273926,   -0.19905543327331543, -1.0114398002624512,  -0.19905543327331543,
        0.207136869430542,    1.0195212364196777,   -0.6052474975585938,  -0.6052474975585938,  -0.6052474975585938,
        1.0195212364196777,   -0.6052474975585938,  -0.19905543327331543, 1.0195212364196777,   0.207136869430542,
        0.207136869430542,    -0.19905543327331543, -1.0114398002624512,  -1.417631983757019,   -1.0114398002624512,
        0.6133291721343994,   1.0195212364196777,   0.207136869430542,    -1.0114398002624512,  -1.0114398002624512,
        -0.6052474975585938,  -2.6362085342407227,  1.0195212364196777,   1.0195212364196777,   -1.417631983757019,
        -1.0114398002624512,  -1.823824167251587,   -1.417631983757019,   -1.417631983757019,   -0.6052474975585938,
        -0.6052474975585938,  1.4257135391235352,   -3.04240083694458,    0.207136869430542,    -0.19905543327331543,
        -0.6052474975585938,  1.0195212364196777,   1.8319058418273926,   -1.417631983757019,   1.8319058418273926,
        -0.19905543327331543, 1.4257135391235352,   -0.6052474975585938,  -0.6052474975585938,  -0.19905543327331543,
        -0.6052474975585938,  0.207136869430542,    -1.417631983757019,   0.207136869430542,    0.207136869430542,
        -0.6052474975585938,  0.6133291721343994,   1.4257135391235352,   -0.19905543327331543, -0.6052474975585938,
        1.0195212364196777,   -0.6052474975585938,  1.4257135391235352,   -0.19905543327331543, -1.0114398002624512,
        1.0195212364196777,   -1.417631983757019,   -0.6052474975585938,  -0.6052474975585938,  -0.6052474975585938,
        1.4257135391235352,   -1.0114398002624512,  0.6133291721343994,   0.6133291721343994,   2.23809814453125,
        -1.417631983757019,   -0.19905543327331543, -1.0114398002624512,  0.207136869430542,    -0.19905543327331543,
        -0.6052474975585938,  -0.6052474975585938,  -1.0114398002624512,  -0.6052474975585938,  0.207136869430542,
        -0.19905543327331543, 0.207136869430542,    -1.417631983757019,   -1.417631983757019,   -0.6052474975585938,
        -0.6052474975585938,  0.6133291721343994,   1.0195212364196777,   -1.823824167251587,   0.6133291721343994,
        -0.6052474975585938,  0.207136869430542,    -1.417631983757019,   -0.19905543327331543, -0.6052474975585938,
        0.207136869430542,    -1.417631983757019,   -0.19905543327331543, -1.823824167251587,   -0.6052474975585938,
        0.207136869430542,    2.644289970397949,    -1.417631983757019,   -1.0114398002624512,  -0.19905543327331543,
        -1.0114398002624512,  1.8319058418273926,   -1.823824167251587,   -1.417631983757019,   -0.6052474975585938,
        1.0195212364196777,   -0.6052474975585938,  0.6133291721343994,   -0.6052474975585938,  -1.0114398002624512,
        -0.19905543327331543, 0.207136869430542,    0.6133291721343994,   -0.6052474975585938,  0.6133291721343994,
        0.6133291721343994,   0.6133291721343994,   -1.417631983757019,   1.0195212364196777,   0.6133291721343994,
        0.6133291721343994,   -0.6052474975585938,  1.4257135391235352,   -0.19905543327331543, 1.0195212364196777,
        -1.0114398002624512,  1.8319058418273926,   0.207136869430542,    -1.417631983757019,   0.207136869430542,
        0.207136869430542,    -0.19905543327331543, 1.0195212364196777,   -0.6052474975585938,  -1.0114398002624512,
        -2.2300164699554443,  -1.417631983757019,   -1.417631983757019,   0.207136869430542,    1.4257135391235352,
        1.0195212364196777,   -0.6052474975585938,  -1.0114398002624512,  1.0195212364196777,   0.207136869430542,
        -0.19905543327331543, 1.4257135391235352,   -1.0114398002624512,  1.0195212364196777,   -0.6052474975585938,
        1.0195212364196777,   -0.19905543327331543, 0.207136869430542,    -0.6052474975585938,  -0.6052474975585938,
        -0.6052474975585938,  -1.0114398002624512,  -0.19905543327331543, -1.0114398002624512,  -0.19905543327331543,
        0.6133291721343994,   -0.6052474975585938,  -0.6052474975585938,  1.0195212364196777,   -1.0114398002624512,
        1.0195212364196777,   -1.823824167251587,   -2.6362085342407227,  -1.0114398002624512,  -1.417631983757019,
        -0.19905543327331543, -0.6052474975585938,  1.4257135391235352,   -1.417631983757019,   0.207136869430542,
        1.4257135391235352,   0.6133291721343994,   -1.0114398002624512,  1.0195212364196777,   0.6133291721343994,
        -0.19905543327331543, -1.823824167251587,   0.207136869430542,    1.0195212364196777,   -0.19905543327331543,
        -0.6052474975585938,  -0.6052474975585938,  0.207136869430542,    1.8319058418273926,   -1.0114398002624512,
        0.6133291721343994,   1.0195212364196777,   -0.19905543327331543, -1.417631983757019,   -0.6052474975585938,
        -0.6052474975585938,  0.207136869430542,    0.6133291721343994,   -1.0114398002624512,  -1.823824167251587,
        -1.823824167251587,   -1.0114398002624512,  1.0195212364196777,   0.6133291721343994,   1.0195212364196777,
        0.6133291721343994,   -1.0114398002624512,  1.0195212364196777,   1.0195212364196777,   0.207136869430542,
        0.207136869430542,    -1.0114398002624512,  1.8319058418273926,   -1.417631983757019,   -0.19905543327331543,
        -0.6052474975585938,  -1.0114398002624512,  0.207136869430542,    0.207136869430542,    -1.417631983757019,
        -0.19905543327331543, -0.19905543327331543, -0.19905543327331543, -0.19905543327331543, -1.417631983757019,
        -1.823824167251587,   -1.0114398002624512,  0.207136869430542,    -1.417631983757019,   0.207136869430542,
        -2.6362085342407227,  -1.823824167251587,   -1.417631983757019,   0.6133291721343994,   0.6133291721343994,
        -0.6052474975585938,  2.23809814453125,     -0.6052474975585938,  0.6133291721343994,   1.0195212364196777,
        -0.19905543327331543, -0.6052474975585938,  0.6133291721343994,   0.6133291721343994,   -0.19905543327331543,
        -1.417631983757019,   -0.19905543327331543, 1.4257135391235352,   -0.6052474975585938,  -1.0114398002624512,
        -0.6052474975585938,  0.207136869430542,    1.4257135391235352,   -1.0114398002624512,  -0.6052474975585938,
        1.0195212364196777,   -1.0114398002624512,  -1.0114398002624512,  -0.6052474975585938,  -0.19905543327331543,
        1.0195212364196777,   1.0195212364196777,   -1.823824167251587,   -0.6052474975585938,  -0.6052474975585938,
        -1.823824167251587,   1.0195212364196777,   -1.0114398002624512,  -0.19905543327331543, -1.0114398002624512,
        -0.6052474975585938,  1.0195212364196777,   0.207136869430542,    0.6133291721343994,   0.207136869430542,
        -1.417631983757019,   1.8319058418273926,   -1.417631983757019,   -0.6052474975585938,  -0.6052474975585938,
        -0.19905543327331543, 0.207136869430542,    0.6133291721343994,   -1.417631983757019,   1.0195212364196777,
        -0.19905543327331543, -1.823824167251587,   0.6133291721343994,   0.6133291721343994,   -1.823824167251587,
        1.8319058418273926,   0.6133291721343994,   -1.0114398002624512,  -1.823824167251587,   -2.6362085342407227,
        -1.417631983757019,   -1.0114398002624512,  0.6133291721343994,   -0.6052474975585938,  -1.417631983757019,
        -0.19905543327331543, 0.6133291721343994,   1.4257135391235352,   1.4257135391235352,   0.207136869430542,
        0.207136869430542,    0.207136869430542,    0.207136869430542,    -0.6052474975585938,  0.207136869430542,
        0.207136869430542,    0.6133291721343994,   -1.417631983757019,   -0.6052474975585938,  1.0195212364196777,
        -1.417631983757019,   1.4257135391235352,   -0.19905543327331543, 1.0195212364196777,   0.6133291721343994,
        1.8319058418273926,   -1.417631983757019,   0.207136869430542,    -1.417631983757019,   0.6133291721343994,
        -0.19905543327331543, -1.0114398002624512,  -0.6052474975585938,  -1.0114398002624512,  -1.0114398002624512,
        0.207136869430542,    -0.19905543327331543, 0.207136869430542,    -1.823824167251587,   -2.2300164699554443,
        -1.0114398002624512,  -1.0114398002624512,  0.6133291721343994,   0.6133291721343994,   -1.823824167251587,
        0.6133291721343994,   0.207136869430542,    0.207136869430542,    -2.2300164699554443,  -0.19905543327331543,
        -0.6052474975585938,  0.6133291721343994,   -1.0114398002624512,  -1.0114398002624512,  -1.823824167251587,
        -0.6052474975585938,  0.207136869430542,    3.0504822731018066,   -1.417631983757019,   -1.417631983757019,
        -0.19905543327331543, -1.417631983757019,   1.0195212364196777,   -1.417631983757019,   -1.417631983757019,
        -0.6052474975585938,  1.0195212364196777,   -0.6052474975585938,  0.207136869430542,    -0.19905543327331543,
        -1.0114398002624512,  -0.19905543327331543, 0.6133291721343994,   0.6133291721343994,   0.207136869430542,
        0.6133291721343994,   -0.19905543327331543, 0.207136869430542,    -1.0114398002624512,  1.4257135391235352,
        0.6133291721343994,   1.0195212364196777,   -0.6052474975585938,  0.6133291721343994,   0.207136869430542,
        1.0195212364196777,   -0.6052474975585938,  0.207136869430542,    0.6133291721343994,   2.23809814453125,
        -1.823824167251587,   -0.6052474975585938,  -0.6052474975585938,  0.207136869430542,    -0.19905543327331543,
        -0.6052474975585938,  -0.6052474975585938,  -0.6052474975585938,  -1.0114398002624512,  0.6133291721343994,
        -0.6052474975585938,  -1.0114398002624512,  -2.2300164699554443,  -0.6052474975585938,  0.207136869430542,
        -0.19905543327331543, 0.207136869430542,    0.207136869430542,    -1.417631983757019,   1.4257135391235352,
        -1.0114398002624512,  -0.19905543327331543, -1.823824167251587,   -0.19905543327331543, -1.0114398002624512,
        0.6133291721343994,   -0.6052474975585938,  0.6133291721343994,   -1.417631983757019,   -0.6052474975585938,
        -0.19905543327331543, 1.8319058418273926,   -1.823824167251587,   -1.417631983757019,   -1.0114398002624512,
        -1.0114398002624512,  0.6133291721343994,   -1.823824167251587,   -1.417631983757019,   -0.19905543327331543,
        1.4257135391235352,   -1.0114398002624512,  -0.19905543327331543, -0.6052474975585938,  -0.6052474975585938,
        1.0195212364196777,   -0.19905543327331543, 0.207136869430542,    -0.19905543327331543, 1.4257135391235352,
        1.0195212364196777,   -0.19905543327331543, -0.6052474975585938,  2.23809814453125,     0.207136869430542,
        0.207136869430542,    -0.6052474975585938,  1.8319058418273926,   -0.6052474975585938,  1.0195212364196777,
        -0.6052474975585938,  0.207136869430542,    0.6133291721343994,   2.23809814453125,     -1.823824167251587,
        -0.6052474975585938,  -0.6052474975585938,  0.207136869430542,    -0.19905543327331543, -0.6052474975585938,
        -0.6052474975585938,  -0.6052474975585938,  -1.0114398002624512,  0.6133291721343994,   -0.6052474975585938,
        -1.0114398002624512,  -2.2300164699554443,  -0.6052474975585938,  0.207136869430542,    -0.19905543327331543,
        0.207136869430542,    0.207136869430542,    -1.417631983757019,   1.4257135391235352,   -1.0114398002624512,
        -0.19905543327331543, -1.823824167251587,   -0.19905543327331543, -1.0114398002624512,  0.6133291721343994,
        -0.6052474975585938,  0.6133291721343994,   -1.417631983757019,   -0.6052474975585938,  -0.19905543327331543,
        1.8319058418273926,   -1.823824167251587,   -1.417631983757019,   -1.0114398002624512,  -1.0114398002624512,
        0.6133291721343994,   -1.823824167251587,   -1.417631983757019,   -0.19905543327331543, 1.4257135391235352,
        -1.0114398002624512,  -0.19905543327331543, -0.6052474975585938,  -0.6052474975585938,  1.0195212364196777,
        -0.19905543327331543, 0.207136869430542,    -0.19905543327331543, 1.4257135391235352,   1.0195212364196777,
        -0.19905543327331543, -0.6052474975585938,  2.23809814453125,     0.207136869430542,    0.207136869430542,
        -0.6052474975585938,  1.8319058418273926,   -0.6052474975585938,  1.0195212364196777,   -0.6052474975585938,
        0.207136869430542,    0.6133291721343994,   2.23809814453125,     -1.823824167251587,   -0.6052474975585938,
        -0.6052474975585938,  0.207136869430542,    -0.19905543327331543, -0.6052474975585938,  -0.6052474975585938,
        -0.6052474975585938,  -1.0114398002624512,  0.6133291721343994,   -0.6052474975585938,  -1.0114398002624512,
        -2.2300164699554443,  -0.6052474975585938,  0.207136869430542,    -0.19905543327331543, 0.207136869430542,
        0.207136869430542,    -1.417631983757019,   1.4257135391235352,   -1.0114398002624512,  -0.19905543327331543,
        -1.823824167251587,   -0.19905543327331543, -1.0114398002624512,  0.6133291721343994,   -0.6052474975585938,
        0.6133291721343994,   -1.417631983757019,   -0.6052474975585938,  -0.19905543327331543, 1.8319058418273926,
        -1.823824167251587,   -1.417631983757019,   -1.0114398002624512,  -1.0114398002624512,  0.6133291721343994,
        -1.823824167251587,   -1.417631983757019,   -0.19905543327331543, 1.4257135391235352,   -1.0114398002624512,
        -0.19905543327331543, -0.6052474975585938,  -0.6052474975585938,  1.0195212364196777,   -0.19905543327331543,
        0.207136869430542,    -0.19905543327331543, 1.4257135391235352,   1.0195212364196777,   -0.19905543327331543,
        -0.6052474975585938,  2.23809814453125,     0.207136869430542,    0.207136869430542,    -0.6052474975585938,
        1.8319058418273926,   -0.6052474975585938,  1.0195212364196777,   -0.6052474975585938,  0.207136869430542,
        0.6133291721343994,   2.23809814453125,     -1.823824167251587,   -0.6052474975585938,  -0.6052474975585938,
        0.207136869430542,    -0.19905543327331543, -0.6052474975585938,  -0.6052474975585938,  -0.6052474975585938,
        -1.0114398002624512,  0.6133291721343994,   -0.6052474975585938,  -1.0114398002624512,  -1.823824167251587,
        -0.6052474975585938,  0.207136869430542,    -0.19905543327331543, 0.207136869430542,    0.207136869430542,
        -1.417631983757019,   1.4257135391235352,   -1.0114398002624512,  -0.19905543327331543, -1.823824167251587,
        -0.19905543327331543, -1.0114398002624512,  0.6133291721343994,   -0.6052474975585938,  0.6133291721343994,
        -1.417631983757019,   -0.6052474975585938,  -0.19905543327331543, 1.8319058418273926,   -1.823824167251587,
        -1.417631983757019,   -1.0114398002624512,  -1.0114398002624512,  0.6133291721343994,   -1.823824167251587,
        -1.417631983757019,   -0.19905543327331543, 1.4257135391235352,   -1.0114398002624512,  -0.19905543327331543,
        -0.6052474975585938,  -0.6052474975585938,  1.0195212364196777,   -0.19905543327331543, 0.207136869430542,
        -0.19905543327331543, 1.4257135391235352,   1.0195212364196777,   -0.19905543327331543, -0.6052474975585938,
        2.23809814453125,     0.207136869430542,    0.207136869430542,    -0.6052474975585938,  1.8319058418273926,
        -0.6052474975585938,  0.207136869430542,    1.0195212364196777,   1.0195212364196777,   1.0195212364196777,
        -0.19905543327331543, -0.19905543327331543, 0.6133291721343994,   -0.19905543327331543, 1.8319058418273926,
        -0.6052474975585938,  0.207136869430542,    0.6133291721343994,   -0.19905543327331543, -1.0114398002624512,
        -0.6052474975585938,  1.4257135391235352,   0.6133291721343994,   -1.417631983757019,   -1.0114398002624512,
        -0.19905543327331543, 0.6133291721343994,   0.6133291721343994,   1.0195212364196777,   -1.417631983757019,
        0.6133291721343994,   -1.0114398002624512,  0.6133291721343994,   -1.823824167251587,   -0.6052474975585938,
        -1.417631983757019,   -0.19905543327331543, 0.207136869430542,    -0.6052474975585938,  -1.823824167251587,
        -0.6052474975585938,  -0.19905543327331543, 0.6133291721343994,   -3.04240083694458,    -0.6052474975585938,
        0.6133291721343994,   -1.417631983757019,   1.0195212364196777,   -0.19905543327331543, -0.19905543327331543,
        -0.6052474975585938,  -0.19905543327331543, -0.6052474975585938,  0.207136869430542,    -0.19905543327331543,
        -2.2300164699554443,  -0.19905543327331543, 1.4257135391235352,   -1.417631983757019,   0.6133291721343994,
        0.207136869430542,    0.6133291721343994,   -0.19905543327331543, -2.2300164699554443,  1.0195212364196777,
        1.0195212364196777,   0.207136869430542,    -1.823824167251587,   -1.417631983757019,   0.207136869430542,
        0.207136869430542,    0.6133291721343994,   1.0195212364196777,   1.4257135391235352,   -0.19905543327331543,
        -0.6052474975585938,  0.6133291721343994,   -0.19905543327331543, 1.4257135391235352,   -0.6052474975585938,
        0.207136869430542,    0.6133291721343994,   -0.19905543327331543, -1.417631983757019,   -0.6052474975585938,
        1.4257135391235352,   0.6133291721343994,   -1.417631983757019,   -1.0114398002624512,  -0.6052474975585938,
        0.207136869430542,    0.6133291721343994,   1.0195212364196777,   -1.417631983757019,   0.6133291721343994,
        -1.0114398002624512,  0.6133291721343994,   -1.417631983757019,   -1.0114398002624512,  -1.417631983757019,
        -0.19905543327331543, 0.207136869430542,    -0.6052474975585938,  -1.823824167251587,   -0.6052474975585938,
        -0.19905543327331543, 1.0195212364196777,   -3.04240083694458,    -0.6052474975585938,  0.6133291721343994,
        -1.417631983757019,   1.0195212364196777,   -0.19905543327331543, -0.19905543327331543, -0.6052474975585938,
        -0.19905543327331543, -1.0114398002624512,  0.6133291721343994,   -0.19905543327331543, -2.2300164699554443,
        -0.6052474975585938,  1.4257135391235352,   -1.417631983757019,   0.6133291721343994,   0.207136869430542,
        0.207136869430542,    -0.19905543327331543, -2.2300164699554443,  1.0195212364196777,   0.6133291721343994,
        0.207136869430542,    -1.823824167251587,   -1.417631983757019,   0.207136869430542,
    };
    const auto result = Embedder->Process("привет, я алиса, очень приятно познакомиться.");
    const auto embeddings = TBertEmbedder::ExtractEmbeddings(result);
    EXPECT_EQ(embeddings.size(), EMBEDDING_SIZE);
    for (size_t i = 0; i < embeddings.size(); ++i) {
        EXPECT_NEAR(embeddings[i], expectedEmbeddings[i], 1e-5);
    }
}

TEST_F(BertEmbedderTest, Edges) {
    const auto queries = TVector<TString>{"", MakeBigQuery(1000)};
    for (const auto& query : queries) {
        const auto result = Embedder->Process(query);
        const auto embeddings = TBertEmbedder::ExtractEmbeddings(result);
        EXPECT_EQ(embeddings.size(), EMBEDDING_SIZE);
    }
}
