#include <alice/beggins/internal/bert_tf/applier.h>

#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/xrange.h>

const TString dataPath = BuildRoot() + "/alice/beggins/internal/bert_tf/ut/data/";
const TString startTriePath = dataPath + "start.trie";
const TString continuationTriePath = dataPath + "cont.trie";
const TString vocabFilePath = dataPath + "vocab.txt";
const TString graphDefPath = dataPath + "model.pb";

using namespace NAlice::NBeggins::NBertTfApplier;

const TUtf32String knownWords = U"сколько стоит новый чайник";
const TVector<TUtf32String> knownWordsTokens{CLS, U"сколько", U"стоит", U"новый", U"чайник", SEP};
const TVector<TUtf32String> knownWordsSplitted{U"сколько", U"стоит", U"новый", U"чайник"};

const TUtf32String unknownWords = U"сколько стоит новенькая 3080ti";
const TVector<TUtf32String> unknownWordsTokens{CLS,    U"сколько", U"стоит", U"нове", U"##нькая",
                                               U"308", U"##0t",    U"##i",   SEP};
const TVector<TUtf32String> unknownWordsSplitted{U"сколько", U"стоит", U"новенькая", U"3080ti"};

const TVector texts = {unknownWords, knownWords};
const TVector textSplits = {unknownWordsSplitted, knownWordsSplitted};
const TVector textTokens = {unknownWordsTokens, knownWordsTokens};

class TBertTfApplierTest : public testing::Test {
public:
    static TVector<TApplier::TResult> Result;

    static void SetUpTestSuite() {
        const TApplier applier(
            std::make_unique<TTokenizer>(TTrie::FromFile(startTriePath), TTrie::FromFile(continuationTriePath)),
            TBertDict::FromFile(vocabFilePath), {.GraphDefPath = graphDefPath,
                                                 .SequenceLength = 40,
                                                 .TokenEmbeddingsOutput = "bert/bert_encoder/encoder/Reshape_9",
                                                 .SentenceEmbeddingOutput = "output_embedding"},
            {.NumInterOpThreads = 16, .NumIntraOpThreads = 16});
        Result = applier.Apply(texts);
    }
};

TVector<TApplier::TResult> TBertTfApplierTest::Result;

TEST_F(TBertTfApplierTest, TestSentenceShapes) {
    for (const auto caseIdx : xrange(2)) {
        ASSERT_EQ(Result[caseIdx].SentenceEmbedding.size(), 1024u);
    }
}

TEST_F(TBertTfApplierTest, TestTokensShapes) {
    for (const auto caseIdx : xrange(2)) {
        ASSERT_EQ(Result[caseIdx].TokenEmbeddings.size(), textTokens[caseIdx].size());
        for (const auto tokenIdx : xrange(Result[caseIdx].TokenEmbeddings.size())) {
            ASSERT_EQ(Result[caseIdx].TokenEmbeddings[tokenIdx].size(), 768u);
        }
    }
}

TEST_F(TBertTfApplierTest, TestWordsShape) {
    for (const auto caseIdx : xrange(2)) {
        ASSERT_EQ(Result[caseIdx].WordEmbeddings.size(), textSplits[caseIdx].size());
        for (const auto wordIdx : xrange(Result[caseIdx].WordEmbeddings.size())) {
            ASSERT_EQ(Result[caseIdx].WordEmbeddings[wordIdx].size(), 768u);
        }
    }
}

TEST_F(TBertTfApplierTest, TestWords) {
    for (const auto caseIdx : xrange(2)) {
        ASSERT_TRUE(Result[caseIdx].Words == textSplits[caseIdx]);
    }
}

const TEmbedding expectedSentenceEmbedding = {
    -1.085,     1.28271,    0.329492,    -0.251583,  -0.750928,   -0.0521863,  0.32719,    0.634583,   -1.69731,
    -0.538688,  2.2654,     0.215308,    0.521724,   1.08646,     -2.15955,    0.271571,   -0.581239,  -0.55904,
    -0.177591,  -0.530343,  0.682941,    0.204386,   1.68486,     -0.531025,   1.5415,     0.539267,   0.782521,
    -1.72809,   -1.4372,    -0.152607,   -0.70223,   -1.181,      -0.616323,   -0.676921,  0.479172,   -0.882377,
    0.537926,   -2.85457,   0.301505,    0.577019,   -0.806608,   0.457341,    1.41734,    -0.217778,  0.595894,
    0.574053,   0.157548,   1.51175,     0.87275,    -0.870246,   -0.478342,   -0.0259755, -0.637367,  0.483112,
    2.03338,    -0.444133,  -1.01703,    -0.338193,  -1.37774,    1.56045,     0.375678,   0.34434,    1.21392,
    -0.738045,  -0.106653,  0.263247,    -0.14096,   1.21562,     -0.673242,   -0.831342,  0.336076,   -0.399304,
    -0.936131,  0.685359,   -1.47138,    -0.703635,  1.12422,     1.55507,     -2.09582,   -0.439462,  -0.428905,
    -0.451875,  -0.518175,  -0.598185,   2.42165,    0.331344,    0.242358,    -1.59124,   -0.221282,  0.318133,
    2.1437,     0.00141997, -1.11091,    0.534681,   0.798168,    -1.39161,    -0.170754,  -0.0638782, 0.0738469,
    0.797587,   0.775335,   0.284594,    0.577945,   -0.121858,   0.463147,    0.0116342,  -0.913845,  -1.64812,
    0.713612,   0.695101,   -1.12835,    0.967016,   2.69317,     -0.782791,   -0.331295,  0.38269,    0.160005,
    -0.976697,  -0.406571,  0.789094,    0.656139,   -1.53631,    -1.86611,    1.88638,    -0.806369,  0.828851,
    -0.484747,  0.313639,   -0.174259,   0.365974,   -0.591075,   0.949823,    -0.707744,  -0.76556,   0.18969,
    -0.287001,  -0.781252,  0.631514,    -1.84541,   -0.368594,   0.818348,    1.25543,    -2.20127,   -0.834256,
    -0.411516,  -0.167497,  -0.259209,   -0.0686739, 2.25622,     0.791734,    0.603255,   -1.89856,   -0.359769,
    0.18397,    1.83128,    -0.170691,   -1.36989,   1.0114,      0.745315,    -1.39605,   0.139283,   -0.220416,
    0.220293,   0.901183,   0.49034,     0.217696,   1.05984,     -0.618886,   0.658435,   -0.0153103, -0.86396,
    -1.62801,   0.891332,   1.19033,     -1.18366,   1.10838,     2.38787,     -1.40258,   -0.449858,  0.903107,
    0.0478542,  -0.807742,  -0.0743944,  0.487004,   0.340339,    -1.10318,    -1.62444,   1.74387,    -0.7118,
    1.01814,    -0.642063,  0.306527,    1.58336,    -0.00677701, -0.757096,   1.64681,    -1.90177,   -1.23286,
    0.135426,   -0.452572,  -0.46327,    0.721588,   -1.2086,     1.65367,     0.908844,   0.593836,   -0.873254,
    0.150349,   0.611321,   0.837076,    0.23577,    0.723362,    1.06151,     0.943788,   0.157345,   -0.25263,
    0.587246,   1.51501,    1.41063,     -0.246102,  -0.704342,   0.593543,    1.42902,    0.376993,   -0.400393,
    0.9384,     0.454858,   1.24264,     0.13579,    -2.15132,    -0.348764,   0.441859,   -2.23004,   -0.722792,
    -0.0190483, -1.31929,   1.10921,     -0.748868,  -1.74558,    -0.291758,   0.168801,   -1.19792,   -0.0682672,
    1.20396,    -0.0699727, -0.746073,   0.313236,   -0.467048,   0.768094,    -1.14471,   -1.45947,   1.22467,
    -1.86872,   -1.06865,   0.73301,     -0.77461,   1.71796,     -0.850923,   -0.238313,  -0.0886779, 1.43219,
    -1.14106,   -0.589259,  -0.174388,   0.311854,   0.745528,    -0.588846,   -1.62521,   0.126402,   -1.01407,
    -1.48083,   -1.50974,   0.574722,    0.13643,    -0.850608,   1.03869,     0.0361551,  0.168868,   0.799081,
    1.21195,    -1.29099,   -0.843625,   -2.1705,    0.162872,    0.269648,    -0.314412,  1.14724,    -1.03217,
    1.43857,    -0.706361,  0.237414,    -0.211401,  1.99808,     -0.337047,   1.96551,    0.187103,   -0.404008,
    0.798048,   -0.343316,  -0.0868772,  0.5439,     1.20144,     -0.611766,   -1.09401,   1.07077,    0.461629,
    0.813354,   1.42212,    0.151993,    -0.445768,  2.86009,     -0.880891,   -0.591602,  -1.60913,   -1.08272,
    0.520253,   -0.892876,  -0.87894,    -0.570582,  0.991825,    0.585717,    -1.04317,   -0.154393,  0.04807,
    3.00983,    -2.03093,   -1.28003,    0.34247,    0.176603,    -0.376795,   -0.181887,  -0.41263,   -0.220423,
    1.13505,    -1.90242,   -0.514803,   0.380415,   0.63775,     0.0599073,   0.60712,    0.44461,    0.359185,
    1.32152,    0.10484,    -0.845099,   -0.298312,  0.0929371,   -1.50028,    -0.312318,  -0.473833,  0.234199,
    1.14713,    0.701716,   -0.550179,   0.913828,   0.74593,     1.93713,     -1.64302,   -1.52683,   -0.69567,
    -1.4577,    0.743107,   -0.0838531,  -0.844843,  0.629669,    2.11749,     -1.32171,   0.0259931,  1.51126,
    -0.0560032, -0.422939,  0.506437,    -0.58663,   -0.704466,   2.31361,     0.0397622,  -0.566507,  -1.73042,
    -0.430978,  0.727638,   0.00624901,  -0.505859,  1.04109,     -0.175182,   0.126,      -0.335388,  0.0597596,
    0.201398,   -0.825917,  0.769849,    -0.579373,  0.694941,    -0.481505,   -0.243682,  0.829496,   0.778446,
    0.374017,   -0.010369,  -0.894397,   1.22568,    0.554708,    -0.72214,    -0.529277,  -0.649303,  -0.896432,
    1.67649,    2.22171,    -0.93861,    1.47309,    -1.7213,     -0.0845218,  -2.52435,   -0.0165351, -0.987745,
    -0.550404,  -1.15682,   0.122445,    -0.87201,   0.467547,    -0.1294,     -0.525076,  -3.05763,   0.556306,
    0.936175,   -1.49412,   1.27429,     1.17828,    0.292876,    1.12369,     1.12519,    0.718534,   1.00042,
    0.606421,   -1.08739,   0.205506,    1.45423,    -1.38743,    0.0472208,   1.31502,    0.50494,    -1.20168,
    -0.720779,  0.766215,   0.484478,    0.418452,   -0.876557,   -0.780347,   0.376319,   0.105738,   -0.236324,
    0.0864718,  0.164355,   -0.710621,   0.633627,   -0.581792,   0.840576,    -0.503064,  -0.31959,   0.682374,
    0.776467,   0.378855,   0.121683,    -1.07036,   1.30509,     0.401933,    -0.583294,  -0.394261,  -0.639133,
    -0.815677,  1.58941,    2.23033,     -0.960697,  1.46831,     -1.67669,    -0.130955,  -2.48866,   0.0678027,
    -1.06784,   -0.611608,  -1.16463,    0.179472,   -0.99479,    0.528982,    -0.270174,  -0.359536,  -3.20745,
    0.33775,    0.922936,   -1.56407,    1.2606,     1.14333,     0.477139,    1.16052,    1.07811,    0.6947,
    1.12611,    0.598297,   -1.14006,    0.126825,   1.2444,      -1.29592,    0.119932,   1.3135,     0.450276,
    -1.2224,    -0.753326,  0.66313,     0.624967,   0.542338,    -0.902999,   -0.588037,  0.470444,   0.0700759,
    -0.241059,  0.301878,   0.156957,    -0.338946,  0.484078,    -0.590503,   1.06036,    -0.415864,  -0.364677,
    0.460585,   0.745721,   0.22852,     0.190527,   -1.1776,     1.30148,     0.393823,   -0.622616,  -0.401703,
    -0.582253,  -0.697425,  1.48106,     2.25181,    -1.06503,    1.36219,     -1.48489,   0.10605,    -2.46488,
    0.180666,   -1.15511,   -0.840231,   -0.864868,  0.146145,    -0.992179,   0.548068,   -0.383928,  -0.156413,
    -3.2299,    0.253558,   1.06391,     -1.6919,    1.42507,     1.15444,     0.711739,   1.10946,    1.01134,
    0.615798,   1.20772,    0.405229,    -1.23186,   -0.0633849,  0.981633,    -1.14004,   0.0839659,  1.31316,
    0.372863,   -1.38199,   -0.908467,   0.584309,   0.677773,    0.743728,    -1.01122,   -0.540132,  0.517184,
    0.217576,   -0.253567,  0.287608,    0.292905,   -0.353855,   0.344976,    -0.633603,  1.07201,    -0.310587,
    -0.326912,  0.506317,   0.796764,    0.19134,    0.104022,    -1.11371,    1.37804,    0.51718,    -0.681027,
    -0.383593,  -0.570245,  -0.781303,   1.33399,    2.25342,     -0.957914,   1.31137,    -1.53905,   0.161237,
    -2.45999,   0.00153667, -1.04259,    -0.769744,  -0.98303,    0.33464,     -1.06861,   0.671934,   -0.288339,
    -0.398218,  -3.24041,   0.399677,    1.10018,    -1.70695,    1.47207,     1.00838,    0.799936,   0.999216,
    0.94063,    0.628882,   1.234,       0.437877,   -1.33118,    -0.00778457, 1.1673,     -1.12651,   0.0182439,
    1.3248,     0.200148,   -1.36039,    -0.846229,  0.71796,     0.614029,    0.534931,   -0.991129,  -0.534689,
    0.350803,   0.0617753,  -0.157209,   0.350536,   0.206298,    -0.349471,   0.334579,   -0.564749,  1.07707,
    -0.275014,  -0.472962,  0.454434,    0.872073,   0.150909,    0.139825,    -1.11936,   1.46373,    0.400281,
    -0.698279,  -0.44847,   -0.637303,   -0.696423,  1.16798,     2.20248,     -0.971396,  1.36675,    -1.47872,
    0.177002,   -2.49344,   -0.0536126,  -0.973509,  -0.817975,   -0.908988,   0.289542,   -1.03601,   0.742028,
    -0.315113,  -0.228278,  -3.30206,    0.416385,   1.12158,     -1.68292,    1.58177,    1.14455,    0.804284,
    0.916279,   0.859761,   0.686812,    1.20832,    0.441356,    -1.3392,     -0.0403425, 1.048,      -1.14318,
    0.083224,   1.27803,    0.21463,     -1.34607,   -0.92093,    0.616887,    0.665658,   0.597647,   -1.01433,
    -0.438871,  0.447899,   -0.00227363, -0.0694613, 0.266654,    0.231817,    -0.638264,  0.352168,   -0.673768,
    0.921212,   -0.253716,  -0.494593,   0.546724,   0.887176,    0.257259,    0.143958,   -1.09627,   1.40956,
    0.408507,   -0.658086,  -0.296601,   -0.816558,  -0.668467,   1.15937,     2.28134,    -0.880125,  1.40389,
    -1.51488,   0.156939,   -2.51964,    -0.247657,  -0.80639,    -0.873815,   -0.990116,  0.218893,   -0.966151,
    0.70221,    -0.269327,  -0.227999,   -3.35613,   0.538826,    1.08991,     -1.46707,   1.58806,    1.17122,
    0.850191,   0.880674,   0.876802,    0.671098,   1.21021,     0.449031,    -1.14798,   -0.0112689, 1.07753,
    -1.18432,   0.10451,    1.35614,     0.136725,   -1.25624,    -1.01057,    0.697607,   0.6213,     0.615503,
    -0.99318,   -0.568061,  0.346615,    0.124771,   -0.082129,   -0.0220502,  0.434941,   -0.857334,  0.426818,
    -0.816708,  0.784922,   -0.235551,   -0.522478,  0.679924,    1.01942,     0.271889,   0.150224,   -1.05465,
    1.38929,    0.39717,    -0.668618,   -0.296576,  -0.849051,   -0.720225,   1.1859,     2.27986,    -0.80605,
    1.38997,    -1.81257,   -0.0180516,  -2.46765,   -0.320233,   -0.600246,   -0.658375,  -1.22016,   0.353882,
    -0.923096,  0.830142,   -0.125173,   -0.36234,   -3.31424,    0.714785,    0.986629,   -1.38523,   1.49481,
    1.0836,     0.849021,   0.873048,    0.846906,   0.660621,    1.10002,     0.560015,   -1.05031,   0.126133,
    1.42203,    -1.29918,   0.125797,    1.24393,    0.216752,    -1.03467,    -0.831746,  0.864217,   0.348116,
    0.386711,   -0.890764,  -0.83799,    0.127829,   0.117591,    -0.0413108,  0.10176,    0.263445,   -0.910469,
    0.436805,   -0.904263,  0.749917,    -0.408286,  -0.472439,   0.724877,    1.02278,    0.471269,   0.160251,
    -1.09167,   1.35056,    0.324125,    -0.650442,  -0.256973,   -0.966938,   -0.662513,  1.07429,    2.27265,
    -0.767641,  1.55115,    -1.59893,    -0.166934,  -2.50864,    -0.416651,   -0.559881,  -0.723695,  -1.41509,
    0.287618,   -0.923085,  0.618427,    -0.28762,   -0.186373,   -3.33722,    0.795378,   1.07435,    -1.22225,
    1.46309,    1.16466,    0.917284,    0.857594,   0.836948,    0.743056,    1.09973,    0.490518,   -1.04868,
    0.119239,   1.23831,    -1.24805,    0.126515,   1.36826,     0.145975,    -1.07201,   -0.804902,  0.768995,
    0.494218,   0.42339,    -0.854946,   -0.728688,  0.254623,    0.255489,    -0.174547,  0.0911987,  0.488123,
    -0.777279,  0.529728,   -0.850933,   0.702124,   -0.275566,   -0.473062,   0.727052,   0.935073,   0.401072,
    0.0192814,  -1.23678,   1.39085,     0.425045,   -0.727824,   -0.174857,   -0.902918,  -0.565641,  1.21996,
    2.2357,     -0.841478,  1.47798,     -1.69151,   -0.00794332, -2.3283,     -0.19808,   -0.61296,   -0.563722,
    -1.40034,   0.458913,   -0.789918,   0.577648,   -0.24152,    -0.272172,   -3.42609,   0.722826,   1.06783,
    -1.23664,   1.50984,    0.919858,    1.02906,    0.937614,    0.912877,    0.510092,   1.03661,    0.537597,
    -1.17546,   0.154423,   1.37251,     -1.24348,   -0.0131713,  1.12197,     0.166581,   -1.14308,   -0.91417,
    0.839442,   0.36458,    0.342076,    -0.915905,  -0.937851,   0.300802,    0.185592,   -0.0794953, 0.000229127,
    0.388786,   -0.762989,  0.541595,    -0.761132,  0.919316,    -0.640243,   -0.547007,  0.738392,   0.857984,
    0.615188,   0.0445782,  -1.34758,    1.22058,    0.363364,    -0.682123,   -0.320672,  -0.984522,  -0.858847,
    1.15854,    2.1239,     -0.627941,   1.59553,    -1.5009,     -0.110551,   -2.53411,   -0.0600179, -0.716063,
    -0.751834,  -1.40775,   0.448278,    -0.893403,  0.717847,    -0.474231,   0.0509875,  -3.41802,   0.736665,
    0.794419,   -1.13151,   1.22748,     0.950217,   0.989413,    1.01222,     0.899373,   0.68626,    1.40272,
    0.578887,   -1.20882,   0.10116,     1.19507,    -1.13799,    -0.00977127, 1.50956,    -0.0792082, -1.10488,
    -0.661687,  0.612396,   0.375406,    0.421984,   -0.724479,   -0.583751,   0.339866}; // сколько стоит новый чайник

const TEmbedding expectedWordEmbedding = {
    -0.165118,  0.997642,   -0.206921,  0.170465,    -0.497381,  -0.561151,   -0.311228,  0.705797,    -0.0856068,
    0.677787,   0.719794,   0.0225313,  -0.13499,    0.422285,   0.363483,    0.371022,   -0.258917,   0.770189,
    0.647385,   -0.290243,  -0.0209548, 0.00449384,  0.783325,   0.424766,    0.17102,    0.576022,    -0.110329,
    0.621069,   0.173249,   -0.0528557, -0.571073,   -0.0867344, 1.21209,     0.708351,   0.273883,    0.161209,
    -0.913555,  0.640303,   -0.102289,  0.507391,    -0.167579,  -0.232101,   -0.471194,  0.128183,    0.227402,
    0.245956,   0.158081,   -1.55204,   0.650239,    0.374317,   -0.0343954,  0.191634,   -0.0633622,  -0.177473,
    0.602914,   -0.0973591, 0.227506,   0.503771,    0.394224,   -0.167343,   0.620426,   0.0450286,   0.272455,
    0.894211,   -0.290818,  -0.431803,  0.325106,    -0.495311,  -0.18189,    -0.109364,  -0.504984,   0.646503,
    -0.42764,   -0.172205,  0.587612,   0.670024,    -0.25503,   0.414664,    -0.957322,  -0.60229,    1.01627,
    -0.0102753, 0.0278151,  -0.231036,  0.196994,    1.01312,    -0.0128001,  0.174307,   -0.35948,    0.265553,
    -0.2047,    0.216199,   0.514396,   -1.84538,    0.274643,   -0.531802,   -0.19576,   -0.30668,    -0.145559,
    -0.592632,  -0.607224,  -0.34295,   -1.22501,    0.370861,   0.142043,    0.839846,   0.665515,    -0.477544,
    -0.473536,  0.117719,   0.00514577, -0.180347,   -0.828388,  -0.515311,   -0.0681162, -0.884456,   -0.200554,
    -0.233453,  0.0337904,  0.467491,   -0.316234,   -0.204383,  -0.0531072,  0.0852395,  0.354431,    0.187538,
    0.0520269,  -0.0486164, 0.756668,   -0.351768,   -0.254416,  0.308827,    -0.129474,  0.0498671,   -0.329385,
    -0.0608731, -0.394073,  -0.835109,  -0.00169187, 1.71184,    0.0330872,   -0.647301,  0.292028,    0.0126411,
    0.48014,    -0.131172,  0.876913,   0.751682,    0.0224031,  0.354051,    0.0530624,  -0.305109,   0.319045,
    0.529577,   -0.351972,  0.294905,   -0.278249,   -0.323794,  1.26027,     -1.50524,   0.984683,    -0.89825,
    -0.314563,  -0.477677,  -0.817616,  1.0123,      0.73021,    -0.443322,   -0.910741,  0.453866,    0.725637,
    -0.490406,  0.395337,   -0.281806,  -0.315687,   0.596927,   -0.535843,   1.27651,    0.944137,    1.01374,
    -0.0972623, 0.771023,   0.829844,   -0.357386,   0.727573,   0.0962672,   0.2773,     1.45574,     -1.73055,
    -0.284756,  0.15669,    -0.22851,   0.0110558,   0.496953,   0.339795,    0.105539,   0.389441,    -0.103838,
    0.531344,   0.509837,   0.389265,   0.00923013,  0.0952599,  -0.0176793,  0.21241,    -0.632697,   0.707255,
    0.00746292, -1.23626,   -0.0176727, -0.348018,   -0.0768954, -0.0525015,  0.191918,   -0.229709,   -0.410366,
    0.557397,   0.269771,   0.177014,   0.15832,     -0.611593,  -0.200308,   0.365442,   -0.178908,   -0.947915,
    -0.237044,  -0.14891,   1.10376,    0.681993,    -0.0623422, 0.434528,    -0.321302,  -0.772226,   0.482039,
    -0.230939,  0.114705,   0.863953,   0.115659,    -1.61095,   1.08318,     0.205893,   -0.194796,   0.443372,
    0.382422,   -1.19887,   -0.481821,  -0.703425,   0.040066,   0.642682,    0.623889,   1.01879,     0.147211,
    0.216902,   -0.472399,  0.553764,   0.592259,    -0.163981,  -0.298012,   0.286309,   0.223646,    -0.0732437,
    -0.602169,  -0.524545,  -0.0942166, 0.138813,    0.2036,     -0.801889,   0.6098,     0.380336,    -0.347445,
    0.133685,   0.702845,   -0.144018,  0.508157,    0.816317,   -0.374277,   0.118328,   -0.561288,   -0.235875,
    -0.19112,   -0.399561,  -0.548442,  0.132533,    0.169047,   -0.0780107,  -0.230944,  0.675637,    -0.213243,
    -0.304136,  0.158793,   -5.02497,   -0.27651,    0.0257961,  0.208349,    0.100562,   -0.516126,   -1.65827,
    0.0304866,  0.167775,   1.92749,    0.56876,     0.546147,   -0.3038,     0.130128,   0.235667,    0.0607102,
    0.153227,   -0.318338,  0.534283,   0.653082,    0.367896,   -0.848644,   -0.307483,  -0.33219,    0.308861,
    -0.136073,  -0.577897,  0.090319,   0.182079,    -0.666862,  -0.112078,   0.548099,   -0.67593,    0.29006,
    -1.24992,   -1.73178,   0.00483298, -0.334478,   0.154342,   0.377558,    1.75489,    -1.57508,    -0.142232,
    -0.450816,  0.149636,   -0.264544,  -1.47433,    0.398401,   0.371916,    0.224521,   -0.013425,   0.0261144,
    -0.353552,  -0.115312,  0.458809,   -0.0742284,  -0.478729,  0.404481,    -1.20434,   0.571516,    -0.358203,
    2.03579,    0.0671823,  -0.188723,  -0.245493,   -0.534911,  -1.41201,    -0.578915,  0.0709828,   0.153189,
    -0.640874,  -0.223864,  -0.0437918, 0.303005,    -0.434425,  0.108061,    -0.414023,  0.190844,    -0.625588,
    -0.665363,  0.0618156,  0.0106974,  0.251192,    -0.418493,  0.371092,    0.412428,   0.948091,    -0.5913,
    -0.360477,  0.0951765,  -0.103867,  0.839996,    0.867256,   0.00641137,  1.00266,    -0.379285,   -0.143924,
    -1.44193,   -0.497607,  0.332548,   0.256307,    -0.0784234, 0.0581536,   -1.01498,   -0.925247,   0.305723,
    -0.561398,  0.360286,   -0.479249,  -0.887939,   0.469359,   0.802893,    0.12652,    1.58874,     0.632404,
    1.03496,    -0.0874852, 0.0123675,  -0.978734,   0.568022,   -0.794427,   -0.563595,  0.109178,    -0.0863795,
    0.289726,   0.489095,   -0.231871,  0.126478,    -1.05917,   -1.02457,    -0.954821,  0.465387,    0.635499,
    0.343213,   0.514158,   -0.511836,  0.431351,    -0.383175,  -0.0675409,  -0.328876,  -0.157818,   -1.1002,
    -0.370184,  0.955563,   -0.514529,  -0.0336955,  -0.0802149, 0.0044121,   -0.118945,  -0.190367,   -0.414643,
    0.264309,   -0.363258,  -0.547406,  -0.59159,    -1.52706,   -0.0902015,  0.00336053, 1.14896,     -0.731861,
    0.958829,   -0.404366,  -1.56263,   -0.266335,   0.405614,   -0.325078,   0.814908,   -0.672119,   -0.753167,
    -1.54934,   -0.31452,   -2.51976,   0.683414,    0.0525429,  -0.117277,   0.0745747,  -2.06391,    0.317691,
    0.125461,   -0.327777,  0.915836,   -0.581433,   -0.424213,  -0.0517156,  0.520877,   0.759656,    -0.176823,
    -0.697849,  0.579803,   -0.211669,  -1.09749,    0.408185,   0.379586,    -0.598764,  0.4001,      -0.0234743,
    -0.211776,  -0.212051,  -0.975635,  -0.572136,   0.581474,   -2.32327,    -0.249531,  0.3291,      1.31383,
    0.108524,   -0.0267255, 1.05487,    0.349617,    1.084,      -0.131336,   0.254995,   0.451047,    0.801167,
    -0.249516,  0.333907,   -0.9032,    -0.809856,   0.802973,   -0.342568,   -0.152492,  -1.67637,    -0.687172,
    -1.04329,   0.768999,   0.643516,   -0.633151,   0.310171,   -0.559702,   -1.06319,   -0.733493,   -0.300771,
    -0.269318,  0.4064,     -0.756051,  -0.0188759,  -0.285778,  0.420426,    0.600737,   -0.708023,   0.0455896,
    0.426125,   0.283135,   0.366872,   0.512526,    -0.347386,  0.130039,    -0.333226,  -0.194388,   0.104135,
    -0.358379,  -0.515102,  1.00462,    1.01807,     -0.233077,  0.387087,    -0.089379,  0.327352,    0.269706,
    0.566643,   -0.297502,  0.394549,   -0.109949,   -0.550493,  -0.584064,   -0.71889,   -0.13012,    0.309335,
    0.219928,   -1.24395,   0.538363,   0.0681731,   0.0679967,  -0.652633,   -0.043222,  0.0739275,   -0.145703,
    1.07458,    0.618296,   0.726614,   -0.107911,   -0.421444,  0.140102,    1.38832,    0.0983153,   0.134208,
    -0.186497,  0.094096,   0.392186,   0.944544,    0.393454,   -1.04917,    0.0743813,  1.44047,     -1.09281,
    0.887744,   -0.381812,  -0.213334,  0.34886,     -1.32224,   -0.00978234, -0.0268151, -0.0173551,  -0.411535,
    0.182132,   0.967259,   -0.452167,  -1.50962,    0.0668374,  0.183774,    0.638814,   -0.784417,   0.31548,
    0.251095,   -1.34001,   -1.16722,   -0.0709386,  0.371503,   -0.384449,   0.250997,   0.699142,    -0.12399,
    -0.88251,   0.245746,   -0.307186,  0.533303,    -0.187525,  0.0555097,   -0.979904,  0.339441,    0.0126164,
    0.713456,   -0.576415,  0.332236,   0.509799,    1.11521,    0.393645,    -1.01674,   -0.478619,   -0.0644349,
    -0.169696,  0.718372,   0.467644,   0.525945,    0.0708912,  -1.00209,    1.90807,    -0.543656,   -1.21615,
    -0.72723,   0.354854,   0.99629,    -0.0792479,  -0.0868534, 0.577488,    1.51225,    0.895712,    -0.0594333,
    -0.927531,  -0.206231,  -0.289782,  1.88969,     0.0743817,  0.359952,    0.463585,   -0.45055,    -0.90854,
    1.1232,     -0.0723199, -0.0781432, 0.148424,    -0.209424,  0.294979,    -0.113736,  -1.09256,    0.468662,
    -0.243932,  0.251719,   0.508819,   1.70896,     0.319651,   0.271479,    -0.199607,  -0.0952364,  -0.124977,
    -0.351725,  -1.35738,   -0.340651,  -0.263697,   -0.491845,  -0.428785,   -0.0496001, 0.854072,    0.0476907,
    0.634045,   -0.336909,  0.304109,   -0.0318595,  -0.538066,  1.04619,     -0.985295,  1.16126,     0.0288174,
    -1.37667,   0.533191,   -0.388485,  0.48946,     -0.12243,   0.829845,    0.315862,   8.75816e-05, -1.69676,
    -1.18545,   -0.692528,  -0.171415,  -0.809713,   0.51529,    -0.400932,   0.976432,   -0.414253,   0.417587,
    0.0289994,  -0.0146988, 1.50061,    -0.076623,   0.355368,   -1.3116,     -0.219835,  0.385659,    -0.593144,
    0.506049,   0.335167,   1.16548,    0.40056,     -0.320097,  0.52777,     1.71462,    -1.05069,    -0.343475,
    -0.169973,  0.121121,   0.117872,   -0.114812,   -0.341181,  0.404201,    -0.203387,  0.206221,    -0.300925,
    0.189447,   1.25749,    0.13649,    -0.401162,   0.572191,   -0.199178,   -0.288518,  -0.387931,   -0.531471,
    -0.227966,  0.406041,   0.753824,   -0.241837,   -0.263444,  -0.430537,   0.3422,     -0.419189,   -2.04604,
    0.778117,   0.173902,   0.18548,    -0.0987519,  0.329999,   -0.110163,   -0.115965,  0.111158,    0.239106,
    0.788157,   -0.597894,  -1.11088}; // сколько [стоит] новый чайник

TEST_F(TBertTfApplierTest, TestSentenceEmbedding) {
    for (const auto idx : xrange(1024)) {
        ASSERT_LE(abs(Result[1].SentenceEmbedding[idx] - expectedSentenceEmbedding[idx]), 0.001);
    }
}

TEST_F(TBertTfApplierTest, TestWordEmbedding) {
    for (const auto idx : xrange(768)) {
        ASSERT_LE(abs(Result[1].WordEmbeddings[1][idx] - expectedWordEmbedding[idx]), 0.001);
    }
}
