# NLG

## Что такое NLG

NLG (Natural Language Generation) — способ преобразования результатов работы сценария Hollywood в текстовые и голосовые ответы Алисы.

NLG работает на языке разметки [Jinja2](https://jinja2docs.readthedocs.io/en/stable/) с дополнительными расширениями, которых нет в оригинальной документации к Jinja2. 

Файлы на языке разметки:
* имеют расширение `.nlg`;
* находятся в подкаталогах `scenarios/scenario_name/nlg`;
* обрабатываются специальным компилятором.

## Логика работы сценария с NLG

1. При регистрации сценария подключается генератор NLG:
   ```cpp
   #include <alice/hollywood/library/scenarios/random_number/nlg/register.h>
   ...
   // Традиционный способ Голливуда
   REGISTER_SCENARIO("random_number",
                     AddHandle<TRandomNumberRunHandle>()
                   .SetNlgRegistration(NAlice::NHollywood::NLibrary::NScenarios::NRandomNumber::NNlg::RegisterAll) );

   // При регистрации сценария на Hollywood Framework
   TRandomNumberScenario::TRandomNumberScenario() {
       ...
       SetNlgRegistration(NAlice::NHollywood::NLibrary::NScenarios::NRandomNumber::NNlg::RegisterAll);
     }
   ```
2. Ответ пользователю определяется внутренней бизнес-логикой сценария.
3. Сценарий формирует набор переменных, которые будут доступны в NLG (**контекст**).
   ```cpp
   nlgData.Context["result"] = 117; // Внутри NLG скрипта будет доступа переменная context.result со значением 117
   ```
4. Сценарий определяет **фразу**, которая получит управление в NLG.
   ```cpp
   bodyBuilder.AddRenderedText("random_number", "render_result"); // Отрендерить фразу "render_result"
   ```
5. После окончания работы сценария NLG подготовит финальный ответ в соответствии с фразой и контекстом. Результатом работы NLG является текстовая строка, которая будет напечатана или произнесена Алисой.

### Пример NLG-файла

Простейший пример NLG-файла русского языка для кода выше:

```
{% phrase render_result %}
    Выпало число not_var{{ context.result }}.
{% endphrase %}
```

Результатом работы NLG скрипта является фраза `Выпало число 117`.

В NLG-файле можно указать одновременно несколько фраз, которые сценарий может выбрать в зависимости от условий.

```
{% phrase render_result %}
    Выпало число not_var{{ context.result }}.
{% endphrase %}
{% phrase render_error %}
    Извините, произошла какая-то ошибка.
{% endphrase %}
```
## Типы переменных контекста

Контекст — это способ передачи переменных из логики С++ кода в логику NLG-кода. В предыдущем примере контекст появлялся в строке `context.result`, а передавался в  `nlgData.Context["result"] = 117;`.

NLG умеет работать со следующими типами переменных контекста:

* логические переменные;
* числовые (целые, вещественные) переменные;
* строковые переменные;
* сложные составные типы (например, такие как дата и время или название города). Примеры работы с такими переменными приведены ниже.

Кроме работы с переменными контекста, NLG позволяет создавать и использовать свои локальные переменные:
* Чтобы выяснить, существует ли та или иная переменная в контексте NLG, используйте метод `is defined`:
  ```
  {% if context.City is defined %}
  ```
* Чтобы сконструировать переменную, используйте метод `set`:
  ```
  {% set message = context.source_date %}
  ```
* Чтобы вывести значение переменной в текст или голос, используйте двойные скобки:
  ```
  not_var{{ context.sum }}
  ```

## Функции NLG
Функции NLG, которые помогают сделать текст человекочитаемым:

* выбирают склонения для слов;
* проставляют заглавные буквы в предложениях;
* выводят человекочитаемые фразы, например, «завтра», «через два дня».

Полный список доступных функций NLG см. в разделе [Сводный список функций и методов](functions.md).

Функции NLG записываются через вертикальную черту:
```
not_var{{ 'кубик' | pluralize(context.count, 'nom') }}
```
Функция `pluralize` выведет слово «кубик», «кубика», «кубиков» в зависимости от значения переменной `context.count` — «2 кубика», но «21 кубик».

```
not_var{{ message | attr('text') | capitalize_first }}
```
Здесь вызываются сразу две функции:
1. Содержимое `message` сначала передается в `attr('text')`. Метод `attr` изменяет атрибуты текстового вывода. 
2. Результат, обработанный функцией `attr` передается в `capitalize_first` — метод делает первую букву предложения заглавной. Результат выводится пользователю.


## Дополнительные ключевые слова
В дополнение к стандартному списку ключевых слов Jinja2 NLG предоставляет дополнительные команды, которые используются для улучшения разметки ответов.

### Выбор фразы для ответа

* `chooseline` — используйте для простых однострочных текстов.
  {% cut "Пример chooseline" %}
  ```
  {% chooseline %}
    Кажется, меня сломали.
    Ой, меня уронили.
    Вам удалось найти во мне ошибку.
  {% endchooseline %}
  ```
  {% endcut %}
* `chooseitem` — используйте для больших текстов. Позволяет задать [кастомный вес](functions.md) для каждого из выбираемых пунктов: `{% chooseitem 0.5 %}`. При наличии символов перевода строки `chooseitem` не будет разделять фразу на несколько. 
  {% cut "Пример chooseitem" %}
  ```
  {% chooseitem %}
    Хорошо, я запомнила ваш выбор!
    {% or %}
    Все готово, я все записала!
  {% endchooseitem %}
  ```
  {% endcut %}



### Выбор канала ответа для части фразы

По умолчанию весь выводимый текст выводится одновременно и как текст, и как голос. Ключевые слова `text` и `voice` позволяют выбрать канал ответа для части фразы. Обычно они используются для дополнительной разметки голосовых сообщений ударениями и падежами.

```
{% voice %}
  Это только голос+а в твоей голове. {# Знак + ставит кастомное ударение #}
{% endvoice %}
{% text %}
  Это только голоса в твоей голове.
{% endtext %}
```

Короткий вариант тегов лучше использовать только inline:
```
Это только {% tx %}голос{% etx %}{% vc %}голос+а{% evc %} в твоей голове.
```

Также разметку текста и голоса можно реализовать через через функцию `attr_`.

### Включить в текущий NLG содержимое другого файла
```
{% ext_nlgimport "alice/hollywood/library/common_nlg/suggests_ru.nlg" %}
```

### Макросы
Макросы позволяют структурировать код NLG и задавать свои собственные локальные функции для удобства обработки.

```
{% macro render_error_nogeo(where) %}
  {% chooseline %}
    к сожалению,
    извините,
    простите,
  {% endchooseline %}
    я не могу понять, где это "not_var{{ where }}".
{% endmacro %}
...
  {# Вызов макроса работает также, как и рендеринг обычных переменных #}
  not_var{{ render_error_nogeo(context.location ) | capitalize_first }}
```
