# Синтаксис языка описания грамматик

Гранет ориентирован на обработку запросов к Алисе, в частности на то, чтобы заполнять слоты [формы](../../frames.md) на основе значений распознанных именованных сущностей (время в формате JSON, идентификатор названного плейлиста и т. д.). Ядро частотных запросов обрабатывается ядром грамматик, специально разработанных для стабильного поведения Алисы на основной массе реплик пользователей. Грамматики для периферийных запросов, специфических для конкретных сценариев, описывают разработчики сценариев, но Гранет позволяет удобно дообучать модель с учетом этих грамматик и тестировать результат.

Подход, основанный на конечном наборе правил, остается достаточно гибким благодаря:

- Nonsense Tagger — нейросетевой компонент, обученный для поиска несущественных частей в предложении («доброе утро», «пожалуйста», «Алиса»).
- Интеграции с компонентами, надежно реализующими распознавание различных именованых сущностей (NER, named-entity recognition).
- Дополнительным синтаксическим возможностям грамматик:
  - Разбор фразы без учета порядка слов, например `[мама мыла раму]` описывает как фразу «мама мыла раму», так и фразу «раму мыла мама».
  - Разбор лемм без учета морфологии TODO: пример.
  - Использование синонимов TODO: пример.
  - Вероятностный подход к выбору лучшего варианта разбора.
  - Обратимая фильтрация несущественных частей предложения, благодаря которой можно более агрессивно фильтровать стоп-слова, не боясь сломать разбор реплики в целом. TODO: пример


## Поддерживаемые конструкции

### Hello world

Простейшее описание интента в Granet выглядит так:

```bash
form alice.light_on:
    root: включи свет
```

`alice.light_on` — имя интента (формы),
`root` — корневой элемент (нетерминал) грамматики для формы `alice.light_on`,
`включи свет` — единственное правило элемента `root`.

Толку от такой грамматики мало, она сматчится только на одну фразу: `включи свет`. Поэтому давайте добавим в неё стоп-слова.

### Стоп-слова

```bash
form alice.light_on:
    root: включи свет
    filler: пожалуйста|алиса
```

Такая грамматика уже наложится на `алиса включи пожалуйста свет`.
Можно продолжить пополнять список стоп-слов, но лучше воспользоваться специальным нейросетевым теггером Nonsense Tagger.
Также стоит использовать готовые наборы стоп-слов из [common/filler.grnt](https://a.yandex-team.ru/arc/trunk/arcadia/alice/nlu/data/ru/granet/common/filler.grnt)

```bash
form alice.light_on:
    root: включи свет
    filler:
        $Common.Filler
        $Common.WayToCallAlice
        $nonsense
```

Такая грамматика сможет наложиться на `доброе утро алиса включи пожалуйста свет`.

### Переменные

В Granet имена элементов обязаны начинаться с символа `$`. Для имен элементов желательно использовать CamelCase — так их проще отличить от обычных слов.

Добавим вариации слова `включи`:

{% list tabs %}

- Без переменной

  ```bash
  form alice.light_on:
      root: (можешь включить|включи|зажги) свет
      filler: $nonsense
  ```

- Переменная в пространстве формы

  Значение переменной с оператором ИЛИ:

  ```bash
  form alice.light_on:
      root: $TurnOn свет
      $TurnOn: можешь включить|включи|зажги
      filler: $nonsense
  ```

  Значение через точку с запятой:

  ```bash
  form alice.light_on:
      root: $TurnOn свет
      $TurnOn: можешь включить; включи; зажги
      filler: $nonsense
  ```

  В столбик:

  ```bash
  form alice.light_on:
      root:
          $TurnOn свет
      $TurnOn:
          можешь включить
          включи
          зажги
      filler:
          $nonsense
  ```

- Переменная в пространстве файла

  ```bash
  form alice.light_on:
      root:
          $TurnOn свет
  $TurnOn:
      можешь включить
      включи
      зажги
  filler:
      $nonsense
  ```

{% endlist %}

Грамматики из примеров с переменной полностью эквивалентны. В этих примерах переменная (нетерминал) `$TurnOn` состоит из трех правил: `можешь включить`, `включи` и `зажги`.
В жизни большинство команд (и не только) описаны где-то в [common](https://a.yandex-team.ru/arc/trunk/arcadia/alice/nlu/data/ru/granet/common) нетерминалах, например, `$Common.TurnOn`, старайтесь их переиспольозвать.


### Мешок фраз

В русском языке, отношения между словами описываются в основном с помощью морфологии (а не с помощью порядка слов, как в английском). В результате слова в русском предложении могут идти почти в произвольном порядке: например, `мама мыла раму` и `раму мыла мама`.

Чтобы обрабатывать эту особенность, Granet поддерживает оператор «мешок фраз» (квадратные скобки). Выражения, перечисленные внутри квадратных скобок, могут идти в предложении в произвольном порядке.

Примеры:
- `[a b c]` эквивалентно `a b c | a c b | b a c | b c a | c a b | c b a`.
- `[(a b) c]` эквивалентно `a b c | с a b` (операндами «мешка слов» являются `a b` и `c`).
- `[a b] c` эквивалентно `(a b | b a) c`.

Добавим этот оператор в нашу грамматику:

```bash
form alice.light_on:
    root: [включи свет]
    filler:
        $Common.Filler
        $Common.WayToCallAlice
        $nonsense
```

Такая грамматика наложится на `включи свет` и `свет включи`. Но не наложится на `включи` и `свет включи свет`. Последний пример можно поддержать так:

```bash
form alice.light_on:
    root: [включи+ свет+]
    filler:
        $Common.Filler
        $Common.WayToCallAlice
        $nonsense
```

Благодаря оператору `+`, такая грамматика наложится на `свет включи свет`.

Почему так происходит. Granet поддерживает типичные для регулярных выражений унарные операторы `?`, `+`, `*`, но в квадратных скобках они интерпретируются менее строго: запись `свет+` означает, что слово `свет` может повториться в реплике 1 и более раз, но эти повторы не обязаны быть последовательными.
Важно по возможности использовать `+` для обязательных слов или нетерминалов и `*` для опциональных, в голосовых запросах пользователи иногда задумываются и повторяют при этом часть фразы.

Бывают случаи, когда порядок важен, например, некоторые слова часто могут нести смысл в середине предложения, но быть незначимыми на концах.
Пример с использованием такого filler'а выглядит так:

```bash
form alice.light_on:
    root: $Common.LeftFiller* [включи+ свет+]
    filler:
        $Common.Filler
        $Common.WayToCallAlice
        $nonsense
```

Такая грамматика наложится на `так все включи свет`.

### Сравнение по леммам {#lemma-compare}

Директива `%lemma` значит, что следующие за ней слова должны матчиться без учета морфологии.

```bash
form alice.light_on:
    root: $Common.LeftFiller* [можешь* $TurnOn+ свет+]
    filler:
        $Common.Filler
        $Common.WayToCallAlice
        $nonsense
$TurnOn:
    %lemma
    включи
    вруби
    зажги
```

Такая грамматика будет матчиться на: `включи свет`, `можешь включить свет`, `свет включить`.

Но не будет матчиться на `могу включить свет`, так как действие директивы `%lemma` рапространяется только на правила элемента `$TurnOn`, перечисленные после неё.

Режим сравнения по леммам можно выставить на всё дерево формы:

```bash
form alice.light_on:
    lemma: true
    root: $Common.LeftFiller* [можешь* $TurnOn+ свет+]
    filler:
        $Common.Filler
        $Common.WayToCallAlice
        $nonsense
$TurnOn:
    включи
    вруби
    зажги
```

Такая грамматика будет матчиться и на `можешь включить свет`, и на `могу включить свет`.
Общий `$Common.TurnOn` описан с использованием матчинга по леммам.


### Синонимы {#synonyms}

Директива `%enable_synonyms all` (используется только совместно с `%lemma`) значит, что следующие за ней слова могут быть сматчены на их "синонимы".

```bash
form alice.light_on:
    lemma: true
    root: $Common.LeftFiller* [можешь* $TurnOn+ свет+]
    filler:
        $Common.Filler
        $Common.WayToCallAlice
        $nonsense
$TurnOn:
    %lemma
    %enable_synonyms all
    включи
    вруби
    зажги
```

Такая грамматика будет дополнительно матчиться на `можешь запускать свет`, `включение света`, `свет зажигаться`, `жечь свет`.
Посмотреть синонимы к запросу можно скриптами [debug_form.sh](https://a.yandex-team.ru/arc/trunk/arcadia/alice/nlu/data/ru/test/granet/debug_form.sh)
или [debug_sample.sh](https://a.yandex-team.ru/arc/trunk/arcadia/alice/nlu/data/ru/test/granet/debug_sample.sh) (entities с типом `syn.*`).
Посмотреть синонимы к нетерминалам своей грамматики можно скриптом [debug_synonyms.sh](https://a.yandex-team.ru/arc/trunk/arcadia/alice/nlu/data/ru/test/granet/debug_synonyms.sh).
Общий `$Common.TurnOn` описан с использованием матчинга по синонимам.


### Склонятор

Пример синтаксиса:

```bash
form f:
    root:
        $Do<g:|inf|ipf|pl|pl,ipf>
        (можешь|можете) $Do<g:inf>
$Do:
    включи
    # ...
```

Матчится на:
```bash
включить
включи
включай
включите
включайте
можешь включить
можете включить
```

В выражении `$Do<g:набор-граммем-1|набор-граммем-2|...>` каждый набор граммем определяет форму, в которую нужно просклонять фразы элемента. Результаты склонятора объединяются по или. Один из наборов граммем может быть пустым. То есть:

- `$Do<g:|inf|ipf|pl|pl,ipf>` = `$Do | $Do<g:inf> | $Do<g:ipf> | $Do<g:pl> | $Do<g:pl,ipf>` = `включи | включить | включай | включите | включайте`
- `$Do<g:inf|ipf|pl|pl,ipf>` = `$Do<g:inf> | $Do<g:ipf> | $Do<g:pl> | $Do<g:pl,ipf>` = `включить | включай | включите | включайте`

Названия граммем см. здесь: [kernel/lemmer/dictlib/ccl.cpp](https://a.yandex-team.ru/arc/trunk/arcadia/kernel/lemmer/dictlib/ccl.cpp)


### Пространства имён {#namespaces}

Чтобы не мучиться с выдумыванием имён, можно поместить описание одной переменной внутрь другой:

```bash
$Play:
    $X<g:|inf|ipf|pl|pl,ipf>
    (можешь|можете) $X<g:inf>
    $X:
        включи
        вруби
        запусти
        # ...
```

Внутри `$Play` нетерминал `$X` доступен по имени `$X`, а снаружи — по имени `$Play.X`


### Слоты {#slots}

Теперь добавим в нашу форму слот `location` — где конкретно включить свет:

```bash
form alice.light_on:
    slots:
        location:
            source: $Location  # создать слот location по переменной $Location
            type: string
    root:
        $Common.LeftFiller* [можешь* $Common.TurnOn+ свет+] ($Common.At $Location)*
$Location:
    %lemma
    кухне|спальне|гостиной
filler:
    $Common.Filler
    $Common.WayToCallAlice
    $nonsense
```

Такая грамматика распознает запросы:
- `включи свет` — без слотов.
- `включи свет на кухне` — в слот `location` попадёт строчка `кухне`.
- `включи свет на кухне пожалуйста` — в слот `location` снова попадёт `кухне` (без `пожалуйста`).

Также доступен такой синтаксис захвата слота:

```bash
form alice.light_on:
    slots:
        location:
            type: string
    root:
        что-то темно на 'кухне'(location) сделай что-нибудь
        $Common.LeftFiller* [можешь* $Common.TurnOn+ свет+] $Common.At 'кухне|спальне|гостинной'(location)
filler:
    $Common.Filler
    $Common.WayToCallAlice
    $nonsense
```

### Слоты и стоп-слова

В примере `включи свет на кухне пожалуйста` имеется неоднозначность со словом `пожалуйста`. Его можно вставить внутрь элемента `$Location`, а можно снаружи — на одном уровне с `$Location`. По умолчанию парсер выбирает последний вариант. Но это поведение можно изменить с помощью директивы `%cover_fillers`. Вот так:

```bash
$Location:
    %cover_fillers
    кухне|спальне|гостинной
```

Такой `$Location` захватит все граничащие с ним стоп-слова. В результате слово `пожалуйста` попадёт в слот `location`.

### Wildcard

В текущем варианте `$Location` содержит фиксированный список комнат, поэтому наша грамматика не распознает запрос `включи свет в коридоре`, так как коридора нет в нашем списке. Давайте добавим отдельный слот, в который будут складываться неизвестные названия комнат.

```bash
form alice.light_on:
    slots:
        location:
            source: $Location
            type: string
        unknown_location:
            source: $UnknownLocation
            type: string
    root:
        $Common.LeftFiller* [можешь* $Common.TurnOn+ свет+] ($Common.At ($Location | $UnknownLocation))*
$Location:
    %lemma
    кухне|спальне|гостинной
$UnknownLocation:
    .+
filler:
    $Common.Filler
    $Common.WayToCallAlice
    $nonsense
```

Новый элемент `$UnknownLocation` содержит единственное правило: `.+`. Точка — это специальный токен, который может матчиться на любое слово. Соответственно `.+` означает последовательность из одного или нескольких произвольных слов.

Такая грамматика найдёт следующие слоты:
- `включи свет пожалуйста` — без слотов
- `включи свет на 'кухне'(location) пожалуйста`
- `включи свет в 'коридоре'(unknown_location) пожалуйста`

Обратите внимание на неоднозначность во втором примере. Для него допустимы несколько вариантов разбора:

1. `включи свет на 'кухне'(location) пожалуйста`
2. `включи свет на 'кухне'(unknown_location) пожалуйста`
3. `включи свет на 'кухне пожалуйста'(unknown_location)`

Парсер выберет первый вариант: по умолчанию (без обучения) вероятностная модель устроена так, чтобы побеждал путь, проходящий по менее вариативным элементам грамматики ([подробнее о вероятностной модели](https://wiki.yandex-team.ru/alice/megamind/nlu/granet/probability/). Точка — очень вариативный элемент, поскольку может превратиться в любое слово (если точней, в любое из `(math) e^{10}` слов — сейчас такая константа). Поэтому выигрывает тот вариант, в котором точка использовалась меньше всего.

### Отрицательные примеры

С помощью директивы `%negative` можно указать отрицательные примеры для элемента. Пример:

```bash
form alice.repeat_after_me:
    root:
        повтори за мной $Request
$Request:
    .+  # положительное правило
    %negative
    [.* триста]  # отрицательное правило
    [.* $sys.swear]  # тоже отрицательное правило
```

Такая форма наложится на `повтори за мной раз два три`, но не наложится на `повтори за мной триста`.

Элемент не накладывается на интервал, если при наложении элемента победило отрицательное правило (расположенное после `%negative`). При этом отрицательное правило должно быть не просто возможным, оно должно быть более вероятным (более конкретным), чем положительные правила.

### Типизированные слоты {#defined-slots}

Рассмотрим такую грамматику:

```bash
form alice.alarm_clock_set:
    slots:
        date:
            # Имя встроенного элемента, описывающего сущность sys.date.
            # Имена таких элементов образуются добавлением символа $ к имени сущности.
            # Элемент $sys.date используется далее в грамматике (см. root),
            # поэтому на него можно ссылаться в параметре source.
            # Обратите внимание, параметр source определяет только границы слота.
            source: $sys.date

            # Тип (имя) сущности, значение которой будет записано в слот.
            # Обратите внимание, в названии типа нет символа $.
            type: sys.date

        time:
            # Элемент, по которому опредеделяются границы слота time.
            source: $Time

            # В один слот можно класть значения разных типов.
            # В параметре type задаётся множество допустимых типов и их приоритет.
            # Первый тип ищется в первую очередь.
            type:
                sys.time
                sys.num

            # Критерий принадлежности сущности к слоту:
            #   inside - сущность должна находиться в пределах границ слота (по умолчанию).
            #   overlap - сущность должна пересекаться со слотом.
            #   exact - границы сущности должны совпадать с границами слота.
            matching_type: exact

    filler:
        $Common.At
        $Common.Filler
        $Common.PronounMe
        $Common.WayToCallAlice
        $nonsense
        как всегда|еще один|можешь|нужно|спокойной ночи
    root:
        %lemma
        [$Common.Set* будильник+ $sys.date* $Time+]
        [разбуди+ $sys.date* $Time+]
$Time:
    $sys.time
    $sys.num
```

Такая грамматика сматчится на:
- `алиса поставь мне будильник на завтра в семь утра`
- `поставь пожалуйста будильник на шесть тридцать`
- `поставь будильник на одиннадцать тридцать пять минут`

Здесь активно используются встроенные элементы, описывающие именованные сущности.
- `$sys.time` — сматчится на `семь утра`, `шесть тридцать`, `одиннадцать тридцать пять минут`.
- `$sys.date` — сматчится на `завтра`.
- `$sys.num` — матчится на числа, например на `сто пятьдесят шесть`.
- `$nonsense` — матчится на несущественные участки предложения («пожалуйста», «алиса» и т. п.)

Именованные сущности ищутся до разбора самой формы. Они представляют собой дуги в графе токенов. Каждая такая дуга содержит тип и значение сущности:

```
. алиса  . поставь . будильник . на . завтра  . в . семь  .  утра .
(nonsense)                          (sys.date )   (sys.time ----->)
                                                  (sys.num)
```

На эти дуги матчатся элементы вида `$имя-типа`. Здесь есть небольшая путаница, поэтому уточню:
- `sys.time` (без доллара) — имя типа, оно же имя сущности;
- `$sys.time` (с долларом) — имя нетерминала, который матчится на сущности `sys.time`.

Предназначение сущностей не только в том, чтобы распознать нужные участки предложения, но и в том, чтобы сконвертировать их в значения соответствующих типов. Например, сущность `sys.time`, наложившаяся на `семь утра`, будет иметь значение `{"hours":7,"period":"am"}`. Эти значения складываются в соответствующие слоты.

Для запроса "алиса поставь будильник на завтра в семь утра" значения слотов будут следующими:

```
date: (sys.date, {"days":1,"days_relative":true})
time: (sys.time, {"hours":7,"period":"am"})
```

В Гранете каждый слот хранит массив значений (массив пар тип-значение). В этот массив складываются все сущности, удовлетворяющие параметрам слота `type` и `matching_type`. Если бы мы не задали параметр `matching_type: exact` у слота `time`, то в него также попала бы сущность `sys.num`:

```
date: (sys.date, {"days":1,"days_relative":true})
time: (sys.time, {"hours":7,"period":"am"})
      (sys.num, 7)
```

ВАЖНО! Секция `type` является **обязательной** при описании слота. Если вам достаточно просто подставить в слот строковое значение, покрытое соответствующим слоту нетерминалом, укажите в секции `type` тип `string`.

Нет ничего страшного в том, что в слот попадают "лишние" значения. Можно всегда смотреть только на первое значение — так поступает Megamind. Порядок значений в массиве определяется порядком типов в параметре `type`.

Может возникнуть ситуация, когда сам слот найден, то есть найдены его границы, но не найдено ни одного значения для этого слота. Такой слот не удаляется. Он просто будет иметь пустой массив значений. Далее, уже при конвертации гранетной формы в Semantic Frame Мегамайнда, для такого слота будет создано значение типа `string`.


### Явное указание значения слота {#explicit-slot-value}

Есть два способа задать значение слота формы без использования именованных сущностей.

#### С помощью директив %type и %value {#type-value}

```bash
form alice.theremin:
    slots:
        beat:
            type: custom.beat
            source: $BeatEnum
    root:
        дай звук $BeatEnum

$BeatEnum:
    %lemma

    # Ассоциирует с каждым правилом нетерминала тип "custom.beat".
    # Действует до конца списка правил, либо до следующего %type.
    %type "custom.beat"

    # Ассоциирует с правилами нетерминала значение "piano".
    # Действует до конца списка правил, либо до следующего %value.
    %value "piano"
    пианино             # Тип "custom.beat", значение "piano"
    рояль               # Тип "custom.beat", значение "piano"

    %value "digital_piano"
    [цифровое пианино]  # Тип "custom.beat", значение "digital_piano"

    %value "guitar_chords"
    [гитарные аккорды]  # Тип "custom.beat", значение "guitar_chords"
    [аккорды на гитаре] # Тип "custom.beat", значение "guitar_chords"

    # ...
```

При таком подходе значения слота `beat` будут взяты из дерева разбора.

#### С помощью теггерной разметки {#tagger-markup}

```bash
form alice.theremin:
    slots:
        beat:
            type: custom.beat
    root:
        дай звук 'пианино'(beat:piano)
        дай звук 'рояля'(beat:piano)
        дай звук 'цифрового пианино'(beat:digital_piano)
        дай звук '[гитарных аккордов]'(beat:guitar_chords)
        дай звук '[аккордов на гитаре]'(beat:guitar_chords)
        # ...
```

Полный синтаксис теггерной разметки: `захвати в слот 'этот текст'(slot/type:value) пожалуйста`

Компоненты `type` и `value` — опциональные. При их отсутствии слот заполняется следующим образом:
- Не указаны `type` и `value` — слот заполняется на общих основаниях, описанных в разделе "Типизированные слоты".
- Указан только `value` — в качестве типа берётся первый тип из параметра слота `type`.
- Указан только `type` — для слота ищется значение сущности этого типа.

Если `value` представляет собой какой-то сложный текст, его можно заключить в кавычки.

Используя такой подход, следует иметь в виду, что сущности, описанные таким способом, видны только Гранету. Более того, они видны только в том случае, если разбор формы прошёлся по правилу, с которым ассоциировано значение сущности.

### Типы именованных сущностей {#named-entity-types}

На данный момент поддерживаются следующие сущности:

- `sys.<...>` — типы данных, пришедшие [из VINS](https://doc.yandex-team.ru/alice/alice-dev-guide/concepts/scenario-form.html#new-forms__section-slot-types). Значения этих типов хранятся в виде json. Список типов:
  - `sys.calc` — математическое выражение. Пример: `один плюс двадцать пять` -> `"{1}  плюс  {25}  "`.
  - `sys.currency` — название валюты. Пример: `рублей` -> `"RUR"`.
  - [`sys.date`](../entities/sys.md) — день. Пример: `послезавтра` -> `{"days":2,"days_relative":true}`.
  - [`sys.time`](../entities/sys.md) — время. Пример: `шесть тридцать` -> `{"hours":6,"minutes":30}`.
  - [`sys.datetime`](../entities/sys.md) — день и время. Пример: `четвёртое апреля семь часов тридцать минут` -> `{"days":4,"hours":7,"minutes":30,"months":4}`.
  - `sys.datetime_range` — интервал времени. Пример: `тридцать минут` -> `{"start":{"minutes":0,"minutes_relative":true},"end":{"minutes":30,"minutes_relative":true}}`.
  - [`sys.weekdays`](../entities/sys.md) — ?. Примеры: "в субботу" -> {"repeat":false,"weekdays":[6]}, `вторник` — почему-то не матчится.
  - `sys.units_time` — ?
  - `sys.fio.surname` — фамилия. Пример: `иванова` -> `"иванов"` (не умеет определять род).
  - `sys.fio.name` — имя. Пример: `василию` -> `"василий"`.
  - `sys.fio.patronym` — отчество. Пример: `петровичу` -> `"петровичу"` (не нормализует).
  - `sys.float` — число с плавающей запятой.
  - `sys.geo` — адрес. Пример: `улица зенитчиков двадцать три` -> `{"house":23,"street":"зенитчиков"}`.
  - `sys.num` — целое число. Примеры: `двадцать три` -> `23`, `девяностые` -> `90`.
  - `sys.album`, `sys.artist`, `sys.films_50_filtered`, `sys.films_100_750`, `sys.poi_category_ru`, `sys.site`, `sys.soft`, `sys.swear`, `sys.track` — различные словари.
- `custom.<...>` — распознавание по словарям. Есть два источника таких сущностей:
  - Бегемотное правило CustomEntities. Сущности ищутся по этим словарям: [alice/vins/apps/personal_assistant/personal_assistant/config/scenarios/entities](https://a.yandex-team.ru/arc/trunk/arcadia/alice/vins/apps/personal_assistant/personal_assistant/config/scenarios/entities). О том, как их добавить можно почитать [здесь](../entities/custom.md).
  - Такие сущности можно искать в самом Гранете (см. "Создание именованной сущности").
- `nonsense` — распознавалка неинформативных участков предложения, которые можно выкинуть без изменения смысла предложения.
- `user.<...>` — сущности, зависящие от конретного пользователя. Например: favorites в навигаторе, название фильма в галлерее.
- `ner.<...>` — сущности из поискового Бегемота:
  - `ner.GeoAddr` — адрес и его компоненты, результат работы бегемотовского правила `GeoAddr`.

### Список метасимволов
(Необязательно полный)

| Символ | Назначение                |
| ------ | ------------------------- |
| \#     | Комментирование           |
| ?      | Повторить 0 или 1 раз     |
| \+     | Повторить 1 или более раз |
| \*     | Повторить 0 или более раз |
| .      | Любое слово               |

### Список директив


#### %lemma
[Сравнение без учета морфологии](#lemma-compare).

Синтаксис: `%lemma [on | off]` (по умолчанию `on`).

Пример:

```bash
$PopularFairyTale:
    %lemma
    золушка
    колобок
    %lemma off
    тараканище
```


#### %weight

Принудительно понизить/повысить вес правила, чтобы корректнее разрешать неоднозначности.

Синтаксис: `%weight <число>`

Пример:

```bash
$SwitchOn:
    включи
    %weight 0.01
    вруби
```


#### %type и %value

[TODO описание]

Синтаксис:

- `%type "<тип>"`
- `%value "<значение>"`

Пример:

```bash
$ConfirmationType:
    %type "custom.confirmation"
    %value "yes"
    да | хорошо | давай
    %value "no"
    нет | не | не надо | да не
```


#### %negative и %positive

`%negative` — обозначение отрицательного правила, `%positive` — отмена предыдущего `%negative`.

Примеры:

```bash
$Request:
    .+  # положительное правило
    %negative
    [.* триста]  # отрицательное правило
```

```bash
$Request:
    положительное правило
    %negative
    отрицательное правило
    опять отрицательное правило
    %positive
    снова положительное правило
```


#### %cover_fillers

Указывает, что извлекая значение для слота нужно также захватывать стоп-слова.

Пример:

```bash
form repeat_after_me:
    slots:
        text:
            source: $Text
            type: string
    root:
        повтори за мной $Text
$Text:
    %cover_fillers
    .+
```


#### %fillers

Отключить фильтрацию стоп-слов.

Синтаксис: `%fillers [on | off]` (по умолчанию `on`).

Пример:

```bash
$VeryExactText:
     %fillers off
     exact text
```


#### %include

Вставить содержимое из другого файла. В начало каждой строки добавляется отступ, соответствующий отступу директивы `%include` в грамматике.

Синтаксис: `%include "<путь>"`

Пример:

```bash
$SongName:
    %include "huge_list_of_songs.txt"
```


#### %enable_synonyms и %disable_synonyms

Включить/отключить матчинг с использованием [синонимов](#synonyms).

Синтаксис: `%enable_synonyms all` для включения вместе с `%lemma`, `%disable_synonyms all` для выключения.

Пример:

```bash
$Radio:
    %lemma
    %enable_synonyms all
    радио
```


### Продолжение следует...

Полный справочник по синтаксису пока что находится в процессе написания. Большинство конструкций можно посмотреть в [юнит-тестах](https://a.yandex-team.ru/arc/trunk/arcadia/alice/nlu/granet/lib/ut/).
