package services

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

	"github.com/karlseguin/ccache/v2"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/alice/iot/bulbasaur/controller/experiments"
	"a.yandex-team.ru/alice/iot/bulbasaur/controller/repository"
	"a.yandex-team.ru/alice/iot/bulbasaur/dto/uniproxy"
	"a.yandex-team.ru/alice/iot/bulbasaur/logging"
	"a.yandex-team.ru/alice/iot/bulbasaur/model"
	"a.yandex-team.ru/alice/library/go/requestid"
	"a.yandex-team.ru/alice/library/go/setrace"
	"a.yandex-team.ru/alice/megamind/protos/common"
	"a.yandex-team.ru/alice/protos/data/location"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
)

func NewUserService(logger log.Logger, reqIDCache *ccache.Cache, repository repository.IController) *UserService {
	return &UserService{
		logger:     logger,
		reqIDCache: reqIDCache,
		repository: repository,
	}
}

type UserService struct {
	logger     log.Logger
	reqIDCache *ccache.Cache // stores request_id:UserInfo mapping
	repository repository.IController
}

func (s *UserService) ProtoUserInfo(ctx context.Context, user model.User) (*common.TIoTUserInfo, error) {
	failWithError := func(message string, err error) (*common.TIoTUserInfo, error) {
		setrace.InfoLogEvent(
			ctx, s.logger,
			fmt.Sprintf("%s: %v", message, err),
		)
		ctxlog.Warnf(ctx, s.logger, "%s: %v", message, err)
		return nil, xerrors.Errorf("%s: %w", message, err)
	}

	requestID := requestid.GetRequestID(ctx)
	userInfoResponse := uniproxy.NewUserInfoResponse(requestID)

	//cache by requestID to mitigate load generated by `partials`
	reqIDCacheKey := fmt.Sprintf("proto_userinfo:%d:%s", user.ID, requestID)
	if requestID != "" {
		item := s.reqIDCache.Get(reqIDCacheKey)
		if item != nil && !item.Expired() {
			protoUserInfo := item.Value().(*common.TIoTUserInfo)
			setrace.InfoLogEvent(
				ctx, s.logger,
				fmt.Sprintf("using UserInfoResponse from cache with key %s", reqIDCacheKey),
				logging.ProtoJSON("user_info", protoUserInfo),
			)
			return protoUserInfo, nil
		}
	}

	// read user info data
	userInfo, err := s.repository.UserInfoWithPumpkin(ctx, user)
	if err != nil {
		switch {
		case xerrors.Is(err, &model.UnknownUserError{}):
			setrace.InfoLogEvent(
				ctx, s.logger,
				fmt.Sprintf("unknown user {id: %d, login: %s}", user.ID, user.Login),
			)
			// serialize empty RawUserInfo into json string
			rawUserInfo, err := json.Marshal(userInfoResponse)
			if err != nil {
				return failWithError("error in user info response marshalling", err)
			}
			// return empty TIoTUserInfo
			protoUserInfo := &common.TIoTUserInfo{
				Rooms:       []*location.TUserRoom{},
				Groups:      []*location.TUserGroup{},
				Devices:     []*common.TIoTUserInfo_TDevice{},
				Scenarios:   []*common.TIoTUserInfo_TScenario{},
				Colors:      []*common.TIoTUserInfo_TColor{},
				Households:  []*common.TIoTUserInfo_THousehold{},
				RawUserInfo: string(rawUserInfo),
			}
			return protoUserInfo, nil
		default:
			return failWithError("error in reading user info", err)
		}
	}

	// serialize RawUserInfo into json string
	var rawUserInfo []byte
	if !experiments.EmptySmartHomeInfo.IsEnabled(ctx) {
		userInfoResponse.Payload = uniproxy.NewUserInfoView(userInfo)
	}
	rawUserInfo, err = json.Marshal(userInfoResponse)
	if err != nil {
		return failWithError("error in user info response marshalling", err)
	}

	// serialize userInfo into TIoTUserInfo{Payload: proto, RawUserInfo: json-string}
	protoUserInfo := userInfo.ToUserInfoProto(ctx)
	protoUserInfo.RawUserInfo = string(rawUserInfo)
	setrace.InfoLogEvent(ctx, s.logger,
		"TIoTUserInfo",
		logging.ProtoJSON("user_info", protoUserInfo),
	)

	// add protoUserInfo to cache for partials
	if requestID != "" {
		s.reqIDCache.Set(reqIDCacheKey, protoUserInfo, time.Second*5)
	}
	return protoUserInfo, nil
}

func (s *UserService) JSONUserInfo(ctx context.Context, user model.User) (uniproxy.UserInfoResponse, error) {
	//cache by requestID to mitigate load generated by `partials`
	requestID := requestid.GetRequestID(ctx)
	reqIDCacheKey := fmt.Sprintf("userinfo:%d:%s", user.ID, requestID)
	if requestID != "" {
		item := s.reqIDCache.Get(reqIDCacheKey)
		if item != nil && !item.Expired() {
			uiv := item.Value().(uniproxy.UserInfoResponse)
			ctxlog.Debugf(ctx, s.logger, "got data with key %s from cache: %#v", reqIDCacheKey, uiv)
			setrace.InfoLogEvent(
				ctx, s.logger,
				fmt.Sprintf("Using UserInfoResponse from cache with key %s", reqIDCacheKey),
				log.Any("payload", uiv.Payload),
			)
			return uiv, nil
		}
	}

	uiv := uniproxy.NewUserInfoResponse(requestID)
	userInfo, err := s.repository.UserInfoWithPumpkin(ctx, user)
	if err != nil {
		switch {
		case xerrors.Is(err, &model.UnknownUserError{}):
			setrace.InfoLogEvent(
				ctx, s.logger,
				fmt.Sprintf("unknown user {id: %d, login: %s}", user.ID, user.Login),
			)
			return uiv, nil
		default:
			setrace.InfoLogEvent(
				ctx, s.logger,
				fmt.Sprintf("unknown error: %v", err),
			)
			ctxlog.Warnf(ctx, s.logger, "unknown error: %v", err)
			return uniproxy.UserInfoResponse{}, xerrors.Errorf("unknown error: %w", err)
		}
	}

	uiv.Payload = uniproxy.NewUserInfoView(userInfo)
	setrace.InfoLogEvent(
		ctx, s.logger,
		"UserInfoView",
		log.Any("payload", uiv.Payload),
	)

	if requestID != "" {
		s.reqIDCache.Set(reqIDCacheKey, uiv, time.Second*5)
	}
	return uiv, nil
}

func (s *UserService) UserExists(ctx context.Context, user model.User) (bool, error) {
	if _, err := s.repository.SelectUser(ctx, user.ID); err != nil {
		if xerrors.Is(err, &model.UnknownUserError{}) {
			return false, nil
		}
		return false, err
	}
	return true, nil
}
