#include "asr_hypothesis_picker.h"
#include "util.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/yexception.h>
#include <util/string/join.h>


namespace NAlice {

namespace {

using TAsrResults = TVector<TAsrHypothesis>;

template <class T>
void Check(const TStringBuf& badWhat, const T& actual, const T& expected) {
    const auto errorMessage =
            TStringBuilder() << "Bad " << badWhat
            << ". Expected '" << expected
            << "', but got '" << actual << "'.";
    UNIT_ASSERT_C(actual == expected, errorMessage);
}

void CheckBestHypothesis(const TStringBuf& actual, const TStringBuf& expected) {
    Check("best hypothesis", actual, expected);
}

void DoCheckTestCase(const TVector<TAsrHypothesisWords>& asrHypothesesWords, TString correctHypoUtterance = "",
                     const TUsefulContextForReranker& contextForReranker = {})
 {
    correctHypoUtterance = correctHypoUtterance.empty() ? JoinSeq(" ", asrHypothesesWords[0]) : correctHypoUtterance;
    const auto bestHypothesisId = PickBestHypothesis(asrHypothesesWords, contextForReranker);
    auto&& pickedHypoUtterance = JoinSeq(" ", asrHypothesesWords[bestHypothesisId]);
    CheckBestHypothesis(pickedHypoUtterance, correctHypoUtterance);
}

void CheckTestCase(const TVector<TString>& asrHypothesesUtterances, TString correctHypoUtterance = "",
                   const TUsefulContextForReranker& contextForReranker = {})
{
    TVector<TAsrHypothesisWords> asrHypothesesWords;
    for (const auto& utterance : asrHypothesesUtterances) {
        asrHypothesesWords.push_back(SplitString(utterance, " "));
    }
    DoCheckTestCase(asrHypothesesWords, correctHypoUtterance, contextForReranker);
}

}  // namespace

Y_UNIT_TEST_SUITE(AsrHypothesisPickerTests) {
    Y_UNIT_TEST(TestNoHeuristicCanBeApplied) {
        CheckTestCase({"привет алиса", "хей алиса"});
    }

    Y_UNIT_TEST(TestShortVsLong_AddWordsToTheEnd) {
        for (const auto& [asrHypothesesTexts, correctAnswer] : TVector<std::pair<TVector<TString>, TString>>{
            {
                {
                    "включи радио",
                    "включи радио вести",
                },
                "включи радио вести",
            },
            {
                {
                    "включи радио",
                    "давай включи что нибудь",
                    "включи радио вести",
                },
                "включи радио вести",
            },
            {
                {
                    "сколько стоит",
                    "а сколько стоят айфон",
                    "сколько стоит айфон",
                },
                "сколько стоит айфон",
            },
            {
                {
                    "сколько стоит",
                    "сколько стоит айфон",
                    "сколько стоит айпад",
                },
                "сколько стоит айфон",
            },
            {
                {
                    "включи вечеринку",
                    "включи вечеринку ханза",
                    "включи вечеринка",
                    "включи вечеринку ханда",
                    "включи вечеринку ханг",
                    "включи вечером",
                    "включи витаминку",
                    "включи вечера",
                },
                "включи вечеринку ханза",
            },
        }) {
            CheckTestCase(asrHypothesesTexts, correctAnswer);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_AddWordsAnywhere) {
        for (const auto& [asrHypothesesTexts, correctAnswer] : TVector<std::pair<TVector<TString>, TString>>{
            {
                {
                    "включи мультфильм",
                    "а включи мультфильм",
                    "включи инспектор мультфильм",
                    "включи инспектор гаджет пожалуйста",
                    "включи инспектор гаджет мультфильм",
                },
                "включи инспектор гаджет мультфильм"
            },
            {
                {
                    "включи плюс",
                    "включи европу плюс",
                    "включи европу пожалуйста",
                    "включи европу плюс а",
                    "включи топ чарт европу плюс",
                },
                "включи топ чарт европу плюс"
            },
            {
                {
                    "на двадцать процентов",
                    "громче на двадцать процентов",
                },
                "громче на двадцать процентов"
            },
            {
                {
                    "на процентов",
                    "громче на двадцать процентов",
                },
                "громче на двадцать процентов"
            },
            {
                {
                    "включи альбом bluetooth",
                    "включи снуп альбом bluetooth",
                    "включи snoop альбом bluetooth",
                    "включи альбом блютуз",
                    "включи снуп альбом блютуз",
                    "включи sb snoop альбом bluetooth",
                },
                "включи снуп альбом bluetooth"
            },
            {
                {
                    "включи что нибудь",
                    "включи что нибудь а я хотел бы жить на манхеттене",
                    "включи что нибудь а я хотела бы жить на манхеттене",
                    "включи что нибудь а я хотела бы жить на манхэттене",
                    "включи что нибудь а я хотел бы жить на манхэттене",
                    "включи что нибудь а",
                    "включи че нибудь",
                    "включи чего нибудь",
                    "включи что нибудь и",
                    "включи что нибудь не",
                    "включи песню",
                },
                "включи что нибудь а я хотел бы жить на манхеттене"
            },
            {
                {
                    "алиса включи песню",
                    "алиса включи песню бум бум",
                    "алиса включи песню бум",
                    "алиса включи песню бум бум бум",
                    "алиса включи песню boom boom",
                    "алиса включи песню руки бум бум",
                    "алиса включи песню бум бум ты ждал дым",
                    "алиса включи песню бум бум ты ж да дым",
                    "алиса включи песню бум бум ты ждал",
                    "включи песню",
                    "алиса включи",
                    "алиса",
                },
                "алиса включи песню бум бум"
            },
            {
                {
                    "поставь timmy trumpet",
                    "поставь the timmy trumpet",
                    "поставь party timmy trumpet",
                    "поставь part the timmy trumpet",
                    "поставь party the timmy trumpet",
                    "поставь part the die timmy trumpet",
                    "поставь party the die timmy trumpet",
                    "поставь part the dear timmy trumpet",
                    "поставь part the dee timmy trumpet",
                    "поставь party the dear timmy trumpet",
                    "поставь",
                },
                "поставь part the timmy trumpet"
            },
        }) {
            CheckTestCase(asrHypothesesTexts, correctAnswer);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Blacklisted_IoTCommands) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {"включи свет", "включи свет на кухне"},
            {"включи свет", "включи свет нам"},
            {"включи чайник", "включи чайник подожди"},
            {"включи кондиционер", "включи кондиционер в ванной"},
        }) {
            CheckTestCase(asrHypothesesTexts);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Blacklisted_IoTCommandsWithIoTUserInfo) {
        const TUsefulContextForReranker::TIoTInfo ioTInfo{{"включи музыку", "время уборки", "пора убираться"}};
        const TUsefulContextForReranker contextForReranker{.IoTInfo = ioTInfo};
        for (const auto& [asrHypothesesTexts, correctAnswer] : TVector<std::pair<TVector<TString>, TString>>{
            {
                {
                    "включи музыку",
                    "включи мою любимую музыку",
                    "включи музыку песня scientist",
                    "включи музыку это сценарий поэтому мы",
                    "должны выбрать нулевую гипотезу",
                },
                "включи мою любимую музыку"
            },
            {
                {
                    "время уборки",
                    "когда наступит время уборки",
                    "а когда уборка",
                    "когда наступит время уборки урожая",
                },
                "когда наступит время уборки урожая",
            },
            {
                {
                    "пора убираться",
                    "пора убираться отсюда",
                    "пора убираться я ухожу пока",
                },
                "пора убираться отсюда"
            }
        }) {
            CheckTestCase(asrHypothesesTexts, correctAnswer);
            CheckTestCase(asrHypothesesTexts, "", contextForReranker);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Blacklisted_ItemSelectorCommands) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {"алиса включи три", "алиса включи три нет"},
            {"включи номер четыре", "включи номер четыре а нет подожди"},
            {"вруби под номером одиннадцать", "вруби под номером одиннадцать хотя нет"},
            {"номер три", "номер три сапога"},
            {"поставь три", "поставь тридцать три"},
            {"два поставь", "два поста", "два постав", "да поставь", "би два поставь", "давай поставь", "два а поставь"},
        }) {
            CheckTestCase(asrHypothesesTexts, "");
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Blacklisted_ArithmeticCommands) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {
                "два плюс два",
                "два плюс два сложи мне",
                "два плюс сто два",
                "два плюс минус два",
            },
            {
                "сколько будет три разделить на пятьдесят",
                "алиса сколько будет если три разделить на пятьдесят",
                "сколько будет тридцать три разделить на сто пятьдесят",
                "сколько будет минус три плюс один разделить на пятьдесят",
            },
            {
                "тысяча минус семьсот пять",
                "тысячу минус семьсот пять",
                "тысяча минус семьсот сорок пять",
            },
            {
                "шесть умножить на восемь",
                "шестьдесят умножить на восемь",
                "шестнадцать умножить на восемь",
                "шестьдесят шесть умножить на восемь"
            },
            {
                "алиса сколько будет три поделить на четыре",
                "алиса сколько будет три умножить на четыре",
                "алиса сколько будет три разделить на четыре",
                "алиса сколько будет три умножить поделить на четыре",
                "алиса сколько будет три и поделить на четыре",
                "алиса сколько будет три делить на четыре",
                "алиса сколько будет три миллиона поделить на четыре",
                "алиса сколько будет три на поделить на четыре"
            },
        }) {
            CheckTestCase(asrHypothesesTexts, "");
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Blacklisted_ShortCommands) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {
                "стоп",
                "стоп что растет в саду",
                "что растет в саду",
                "запиши что растет в саду",
                "алиса что растет в саду",
            },
            {
                "алиса стоп",
                "а в какое время намаз утренний",
                "алиса стоп а в какое время намаз утренний",
                "алиса стоп а какое время намаз утренний",
                "а какое время намаз утренний",
            },
            {
                "выключай",
                "выключай три мушкетера",
                "выключай алиса три мушкетера",
                "выключай алиса",
                "выключай включи три мушкетера",
                "выключай еще три мушкетера",
            },
            {
                "нет",
                "нет нет",
                "это корова нет",
                "это нет",
                "нет корова нет",
                "это корова",
                "корова нет",
                "это здорово нет",
                "это не корова нет",
                "",
            },
            {
                "отключись алиса",
                "отключись алиса на работу",
                "включись алиса",
                "подключись алиса",
                "подключись алиса на работу",
                "заткнись алиса",
                "отключайся алиса",
                "выключись алиса",
                "включись алиса на работу",
                "отключись алиса нахуй",
                "отключись алиса на работе",
            },
            {
                "продолжай",
                "продолжай alma",
                "продолжай алмаз",
                "продолжай alpha",
                "продолжай алмазы",
                "продолжай а у вас",
                "продолжай а у нас",
                "продолжай алфавит",
                "продолжай колбас",
            },
            {
                "домой",
                "домой они платные",
                "домой а не платный",
                "домой а не платные",
                "давай",
                "домой а не платный бизнес",
                "домой они платные в смысле",
                "домой а не платный смысл",
                "домой а не платные смыслы",
            },
            {
                "предыдущую",
                "предыдущую вот они мои железные денежки",
                "предыдущую потише пожалуйста",
                "предыдущую вот они мои железные денежки потише пожалуйста",
                "предыдущую вот они мои железные денюжки",
                "предыдущую потише",
                "следующую",
                "предыдущая",
                "предыдущий",
                "предыдущую тише пожалуйста",
                "потише пожалуйста",
            },
        }) {
            CheckTestCase(asrHypothesesTexts);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_AdvanceIsNotNeeded_FilterRubbishAndRepeatsInAlternatives) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {
                "включи последние новости",
                "давайте включи последние новости",
                "включи последнюю новость",
                "давай включи последние новости",
                "включи последний новости",
                "включи последняя новость",
                "включи какие новости",
                "включи последнюю новости",
                "включи новости",
                "давайте давайте включи последние новости",
            },
            {
                "алиса нарисуй",
                "алиса нарисуй его",
                "алиса нарисуй луну",
                "алиса нарисуй меня",
                "алиса нарисовала",
                "алиса нарисовано",
                "алиса нарисуй мне",
                "алиса нарисован",
                "алиса нарисую",
                "алиса нарисова",
                "алиса",
                "алиса на рисунок",
            },
            {
                "смотреть",
                "смотреть фильм",
                "смотреть музыку",
                "смотреть фильмы",
                "смотреть видео",
                "смотрите",
                "смотрить",
                "смотреть мультик",
            },
            {
                "включи артем кей песня кис кис кис кис мяу мяу",
                "включи артем кей песня кис кис кис мяу мяу",
                "включи артем кей песня кис кис кис кис мяу мяу мяу",
                "включи артем кей песня кис кис кис кис мяу мяу чу",
                "включи артем кей песня кис кис кис кис мяу мяу что",
                "включи артем кей песня кис кис кис кис мяу мяу че",
                "включи артем кей песня кис кис кис мяу мяу чу",
                "включи артем кей песня кис кис кис кис мяу мяу чо",
            },
            {
                "включи незабудка",
                "включи включи незабудка",
                "включи улыбка",
                "включи витаминка",
                "включи любимка",
                "алиса включи незабудка",
                "включи кукушка",
                "включи медуза",
                "включи белоруска",
            },
            {
                "кто такой бага",
                "кто такой гага",
                "кто такой бага алиса",
                "кто такой бага алиса кто",
                "кто такой гага алиса",
                "кто такой гага алиса кто",
            },
            {
                "поставь семь звук",
                "поставь в семь звук",
                "поставь на семь звук",
                "поставь семь звуки",
                "поставь всем звук",
                "поставь семь звонок",
                "поставь шесть звук",
                "поставь семь",
                "поставь восемь звук",
            },
            {
                "как можно узнать на моем счете сколько денег",
                "как можно узнать на чем на моем счете сколько денег",
                "как можно узнать на счет на моем счете сколько денег",
                "как можно узнать на на моем счете сколько денег",
                "как можно узнать насчет на моем счете сколько денег",
                "как можно узнать на все на моем счете сколько денег",
                "как можно узнать на всем на моем счете сколько денег",
                "как можно узнать на что на моем счете сколько денег",
                "как можно узнать наш на моем счете сколько денег",
                "как можно узнать на сон на моем счете сколько денег",
                "как можно узнать еще на моем счете сколько денег",
                "как можно узнать на счете сколько денег",
                "как можно узнать на своем счете сколько денег",
                "как можно узнать нашел на моем счете сколько денег",
                "как можно узнать на в моем счете сколько денег",
                "как можно узнать на семь на моем счете сколько денег",
                "как можно узнать зачем на моем счете сколько денег",
                "как можно узнать наше на моем счете сколько денег",
                "как можно узнать наши на моем счете сколько денег",
                "как можно узнать на черном счете сколько денег",
            },
            {
                "можно мне понять можно мне понять как в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как в словесном разговоре считается если у нас русских бог",
                "можно мне понять можно мне понять как ну в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как ну в словесном разговоре считается если у нас русских бог",
                "можно мне понять можно мне понять как иници ну в словесном разговоре считается если у нас у русских бог",
                "можно мне понять подожди можно мне понять как в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как иници в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как не в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как инициа в словесном разговоре считается если у нас у русских бог",
                "можно мне понять подожди можно мне понять как в словесном разговоре считается если у нас русских бог",
                "можно мне понять можно мне понять как инесса в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как иници в словесном разговоре считается если у нас русских бог",
                "можно мне понять можно мне понять как иници ну в словесном разговоре считается если у нас русских бог",
                "можно мне понять можно мне понять как инициа в словесном разговоре считается если у нас русских бог",
                "можно мне понять можно мне понять как инициативном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как инеции в словесном разговоре считается если у нас у русских бог",
                "можно мне понять подожди можно мне понять как иници ну в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как иниции в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как инициу в словесном разговоре считается если у нас у русских бог",
                "можно мне понять можно мне понять как в словестном разговоре считается если у нас у русских бог",
            },
            {
                "кто такой жуков",
                "кто тут такой жуков",
                "кто ты такой жуков",
                "кто предпокоит жуков",
                "кто тут тупой жуков",
                "только кто поет жуков",
                "кто тут плохой жуков",
                "кто тут попоет жуков",
                "кто тут попой жуков",
                "кто хоть тупой жуков",
                "кто хоть такой жуков",
                "кто тут покоит жуков",
            },
            {
                "слушать песню",
                "слушать пень песню",
                "слушать пень песня",
                "слушать песни",
                "слушать песня",
                "слушать петь песню",
                "слушать песню а",
                "слушать пен песню",
                "слушать пение песню",
                "слушать спеть песню",
                "слушать пень песни",
                "слушать пень песню а",
                "слушать песню о",
                "слушать",
                "слушать пень <unk> песню",
                "слушать <unk> песню",
                "слушать <unk> песня",
                "слушать песню <unk>",
            },
            {
                "алиса включи",
                "алиса включи музыку",
                "алиса включи мама с викой",
                "алиса включи мама света",
                "алиса включи ютуб",
                "алиса включить",
                "алиса включи нам",
                "алиса включи мне",
                "алиса включи мама",
                "алиса включай",
                "алиса включи как",
                "алиса включи и",
                "алис включи",
                "включи",
                "алиса",
                "алиса <unk> алиса включи",
                "алиса включи <unk>",
                "<unk> алиса включи"
            },
            {
                "что значит слово",
                "что значит слово д",
                "что значит слово з",
                "что значит слово the",
                "что значит слово дэп",
                "что значит слово зет",
                "что значит слово зэ",
                "что значит слово z",
                "что значит слово длб",
                "что значит слово зе",
                "что значит слово зеп",
            },
            {
                "отлично алиса выезд",
                "отлично алиса выезд пол",
                "отлично алиса выезд бол",
                "отлично алиса выезд на",
                "отлично отлично выезд",
                "отлично алиса выезд по",
                "отлично алиса выезд болот",
                "отлично алиса выезд в",
                "отлично алиса выезд поло",
                "отлично алис выезд",
                "ну отлично алиса выезд",
                "отлично алиса выезд к",
            },
            {
                "настоящий телефон для девочек смартфон",
                "настоящий телефон для девочек смартфон настоящий ты что тупая маша",
                "настоящий телефон для девочек смартфон какой настоящий ты что тупая маша",
                "настоящий телефон для девочек смартфон настоящий",
                "настоящий телефон для девочек смартфоны",
                "настоящий телефон для девочек в смартфон",
                "настоящий телефон для девочек смартфон ой настоящий ты что тупая маша",
                "настоящий телефон для девочек смартфон тупая маша",
                "настоящий телефон для девочек смартфона",
                "настоящий телефон для девочек смартфон настоящий ты тупая маша",
                "настоящий телефон для девочек смартфон ты что тупая маша",
            },
            {
                "алиса какие последние новости",
                "алиса какие последние ее новости",
                "алиса какие последние или новости",
                "алиса какие последние и новости",
                "алиса какие последний новости",
                "алиса какие последняя новости",
                "алиса какие последнюю новости",
                "алиса какие последние еве новости",
                "алиса какие последние его новости",
                "алиса какие последние имя новости",
                "алиса какие последние иви новости",
                "алиса какие последние эве новости"
            },
            {
                "я понимаю",
                "я не понимаю",
                "я это понимаю",
                "я тебя понимаю",
                "я так понимаю",
                "я их понимаю",
                "я и понимаю",
                "не понимаю",
                "я то понимаю",
                "я ведь понимаю",
                "нет понимаю",
                "понимаю",
                "блядь понимаю",
                "блять понимаю",
                "я",
                "пять понимаю",
            },
        }) {
            CheckTestCase(asrHypothesesTexts);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_AdvanceIsNotNeeded_TryNotExtendAlreadyLongHypos) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {
                "да вот забыла будет одиннадцать а какая сейчас я прослушала",
                "да вот забыла будет одиннадцать а какая сейчас я прослушала то есть там",
                "да вот забыла будет одиннадцать а какая сейчас я прослушала то есть заново",
                "да вот забыла будет одиннадцать а какая сейчас я прослушала ты самая",
                "да вот забыла будет одиннадцать а какая сейчас я прослушала то есть там его",
            },
            {
                "грибы всасывающие органические вещества живых организмов",
                "лежу от телефона грибы всасывающие органические вещества живых организмов",
                "грибы высасывающие органические вещества живых организмов",
                "грибы всасывающий органические вещества живых организмов",
                "лежу грибы всасывающие органические вещества живых организмов",
                "грибы всасывающие органические вещества в живых организмов",
                "грызут грибы всасывающие органические вещества живых организмов",
                "грыжу от телефона грибы всасывающие органические вещества живых организмов",
                "лишь от телефона грибы всасывающие органические вещества живых организмов",
                "грыжи от телефона грибы всасывающие органические вещества живых организмов",
                "лежит телефон грибы всасывающие органические вещества живых организмов",
            },
        }) {
            CheckTestCase(asrHypothesesTexts);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_AdvanceIsNotNeeded_StricterConditionsToExtendOneWordZeroHypo) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {
                "алиса",
                "алиса поставь",
                "алиса пока",
                "алиса под номером шестнадцать",
                "алиса под",
                "алиса погода",
                "алиса а",
                "алиса хватит",
                "алиса поставь на",
                "алиса под номером",
                "алиса поставь двадцать",
            },
            {
                "алиса",
                "алиса включи",
                "alice",
                "алиса ты",
                "алиса как",
                "алиса стоп",
                "алиса выключи",
                "стоп алиса",
                "адреса",
                "",
                "алисочка",
            },
            {
                "цыганка",
                "цыганская",
                "цыганка пять",
                "цыганка пятая",
                "цыганка пятый",
                "цыганка спят",
                "цыганка с",
                "цыганка пятка",
                "цыганка пят",
                "цыганка в",
                "цыганка как",
                "бьянка",
            },
            {
                "викинги",
                "викинги и существа",
                "викинги существа",
                "викинги существо",
                "викинги существуют",
                "викинги и",
                "викинги сосиска",
                "викинги с",
                "викинг",
                "викинги из",
                "дикие",
            },
            {
                "алиса",
                "алиса включи",
                "алиса вконтакте",
                "алиса алиса",
                "алиса хватит",
                "а алиса",
                "алиса клип",
                "алисов",
                "алиса ой",
                "алиса вк",
                "алиса все",
            },
            {
                "страшилки",
                "машины страшилки",
                "страшные страшилки",
                "смешные страшилки",
                "смешилки",
                "не страшилки",
                "про страшилки",
                "на страшилки",
                "смешинки",
                "смешанные страшилки",
                "мне страшилки",
            },
        }) {
            CheckTestCase(asrHypothesesTexts);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Limitations_DontAddPresentWords) {
        for (const auto& [asrHypothesesTexts, correctAnswer] : TVector<std::pair<TVector<TString>, TString>>{
            {
                {
                    "включи песню все",
                    "включи песню все все"
                },
                "включи песню все",
            },
            {
                {
                    "какое проверочное слово сестра",
                    "какое проверочное слово сестра сестра"
                },
                "какое проверочное слово сестра",
            },
            {
                {
                    "электронов",
                    "или куча электронов",
                    "электро или куча электронов",
                    "электролит куча электронов",
                    "и липучий электронов",
                    "и летучий электронов",
                    "ли куча электронов",
                    "липучий электронов",
                    "электро электронов",
                    "куча электронов",
                    "электролит кучи электронов",
                },
                "или куча электронов",
            },
        }) {
            CheckTestCase(asrHypothesesTexts, correctAnswer);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Limitations_DontAddBeforeFirstAlice) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {
                "алиса что такое тренинг по английскому",
                "алиса что такое тренинг по английски",
                "алиса что такое тренинг на английском",
                "зачем ты перебила алиса что такое тренинг по английскому",
                "зачем ты это перебила алиса что такое тренинг по английскому",
                "зачем ты ее перебила алиса что такое тренинг по английскому",
            },
            {
                "алиса как называется собака по китайски",
                "мимимишки алиса как называется собака по китайски",
                "мимишки алиса как называется собака по китайски",
                "минишки алиса как называется собака по китайски",
                "алиса как называется собакам по китайски",
            },
        }) {
            CheckTestCase(asrHypothesesTexts);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Limitations_DontAllowTooManyRubbishWordsInAlternativeHypo) {
        for (const auto& [asrHypothesesTexts, correctAnswer] : TVector<std::pair<TVector<TString>, TString>>{
            {
                {
                    "алиса а можно такая сказка",
                    "алиса а можно вот такая сказка",
                    "алиса а можно буратино такая сказка",
                    "алиса а можно ботинок такая сказка",
                    "алиса а можно какая нибудь сказка",
                },
                "алиса а можно буратино такая сказка"
            },
            {
                {
                    "хорошо спасибо",
                    "хорошо спасибо яндекс",
                    "спасибо",
                    "спасибо яндекс",
                    "хорошо спасибо поиграем яндекс",
                    "хорошо спасибо поиграем в яндекс",
                    "хорошо",
                    "хорошо спасибо давай поиграем яндекс",
                    "хорошо спасибо давай поиграем в яндекс",
                    "хорошо спасибо в города играем яндекс",
                    "хорошо ну спасибо",
                    "хорошо яндекс",
                },
                "хорошо спасибо поиграем яндекс"
            },
            {
                {
                    "алиса расскажи стих навигатор",
                    "алиса расскажи стих вы навигатор",
                    "алиса расскажи стих в навигатор",
                    "алиса расскажи стих и навигатор",
                    "алиса расскажи стих д навигатор",
                    "алиса расскажи стих да навигатор",
                },
                "алиса расскажи стих навигатор"
            },
        }) {
            CheckTestCase(asrHypothesesTexts, correctAnswer);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Limitations_DontAllowForbiddenWords) {
        for (const auto& [asrHypothesesTexts, correctAnswer] : TVector<std::pair<TVector<TString>, TString>>{
            {
                {
                    "поставь руки вверх",
                    "руки вверх",
                    "поставь клип руки вверх",
                    "включи руки вверх",
                    "поставь мне руки вверх",
                    "поставь группу руки вверх",
                    "поставь мультик руки вверх",
                    "поставь включи руки вверх",
                    "поставь песню руки вверх",
                    "поставь нам руки вверх",
                },
                "поставь руки вверх"
            },
            {
                {
                    "ac dc",
                    "ac dc видео",
                    "ac dc включи",
                    "ac dc фильм вруби плз",
                    "ac dc поставь",
                    "ac dc tnt",
                    "из ac dc",
                },
                "ac dc включи"
            },
            {
                {
                    "включи сказку",
                    "включи сказку иди на хуй",
                    "включи сказку единорог",
                    "включи сказку на ночь",
                    "включи сказку про",
                    "включи сказку иди на",
                    "включи сказку иди нахуй",
                    "включи сказку единорожка",
                    "включи сказку идти на улицу",
                    "включи сказку петина",
                    "включи сказку зима",
                    "включи сказку иди на ночь",
                    "включи сказку вите надо выйти",
                    "включи сказки",
                    "включи",
                },
                "включи сказку единорог",
            },
        }) {
            CheckTestCase(asrHypothesesTexts, correctAnswer);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Limitations_DontAllowSomeAloneWordsToBeAdded) {
        for (const auto& [asrHypothesesTexts, correctAnswer] : TVector<std::pair<TVector<TString>, TString>>{
            {
                {
                    "включи топ чарт",
                    "включи топ чарт музыку",
                    "включи топ чарт европа плюс",
                    "включи топ чарт европы плюс",
                    "включи топ чарт прямой",
                },
                "включи топ чарт европа плюс",
            }
        }) {
            CheckTestCase(asrHypothesesTexts, correctAnswer);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Limitations_DontApplyHeuristicIfMissedTextIsTooLong) {
        for (const auto& [asrHypothesesTexts, correctAnswer] : TVector<std::pair<TVector<TString>, TString>>{
            {
                {
                    "включи плейлист дня",
                    "включи плейлист дня я сам выбрал что это называется у алисы",
                    "включи плейлист дня я сам выйду что ты называешь ее алисой",
                    "включи плейлист дня я сам выбрал что это называется",
                    "включи плейлист дня я сам выбрал что это называется меридиан",
                    "включи плейлист дня я сам выбрал что это называется мелеуз",
                    "включи плей лист дня",
                    "включи плейлист дня я сам выйду что ты называешь",
                },
                "включи плейлист дня",
            },
            {
                {
                    "выключи музыку",
                    "выключи музыку так тогда дети садитесь рядом иди туда папа иди ко мне",
                    "выключи музыку тогда дети садитесь рядом иди туда папа иди ко мне",
                    "выключи музыку давай так тогда дети садитесь рядом иди туда папа иди ко мне",
                    "выключи музыку давай раздавай так тогда дети садитесь рядом иди туда папа иди ко мне",
                    "выключи музыку давайте садитесь рядом иди туда папа иди ко мне",
                    "выключи музыку давай раздавай как тогда дети садитесь рядом иди туда папа иди ко мне",
                    "выключи музыку давай раздавай так тогда дети садитесь рядом иди туда пап иди ко мне",
                },
                "выключи музыку",
            },
            {
                {
                    "включи беловежскую пущу",
                    "включи беловежскую пущу появились довольно четкие жесткие слова о том что",
                    "включи беловежскую пущу четкие жесткие слова о том что",
                    "включи беловежскую пущу довольно четкие жесткие слова о том что",
                    "включи беловежскую пущу появились довольно четкие жесткие слова в том что",
                    "включи беловежскую пущу четкие жесткие слова в том что",
                    "включи беловежскую пущу жесткие слова о том что",
                    "включи беловежскую пущу на смерть обязательное появились довольно четкие жесткие слова о том что",
                    "включи беловежскую пущу на смерть обязательное довольно четкие жесткие слова о том что",
                    "включи беловежскую пущу на сверх обязательное довольно четкие жесткие слова о том что",
                },
                "включи беловежскую пущу",
            },
        }) {
            CheckTestCase(asrHypothesesTexts, correctAnswer);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_Other_DontChangeLastWordEnding) {
        for (const auto& asrHypothesesTexts : TVector<TVector<TString>>{
            {"включи стол", "включи столб"},
            {"включи бра", "включи брат"},
            {"что такое хай", "что такой хайп"},
            {"включи номер два", "включи номер двадцать"},
        }) {
            CheckTestCase(asrHypothesesTexts);
        }
    }

    Y_UNIT_TEST(TestShortVsLong_DontCrushOnEmptyZeroHypo) {
        CheckTestCase({"", "пустая строка в нулевой гипотезе", "не должна крашить всё"});
    }
}

} // namespace NAlice
