# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
import os
import pytest
import yatest.common

from alice.nlu.py_libs.respect_rewriter import RespectRewriter

logger = logging.getLogger(__name__)


@pytest.fixture(scope='module')
def parser_path():
    return yatest.common.binary_path('alice/nlu/py_libs/syntax_parser/model')


@pytest.fixture(scope='module')
def models_path():
    return yatest.common.binary_path('alice/nlu/py_libs/respect_rewriter/models')


@pytest.fixture(scope='module')
def pluralizer(parser_path, models_path):
    return RespectRewriter(
        parser_path,
        to_plural=True,
        to_gender=None,
        classifier_path=os.path.join(models_path, 'classifier_model.tar'),
        embeddings_path=os.path.join(models_path, 'embeddings.tar'),
    )


@pytest.fixture(scope='module')
def pluralizer_with_tagger(parser_path, models_path):
    return RespectRewriter(
        parser_path,
        to_plural=True,
        to_gender=None,
        classifier_path=os.path.join(models_path, 'classifier_model.tar'),
        tagger_path=os.path.join(models_path, 'tagger_model.tar'),
        embeddings_path=os.path.join(models_path, 'embeddings.tar'),
    )


@pytest.fixture(scope='module')
def male_singularizer(parser_path):
    return RespectRewriter(parser_path, to_plural=False, to_gender=RespectRewriter.MASC_GENDER)


@pytest.fixture(scope='module')
def female_singularizer(parser_path):
    return RespectRewriter(parser_path, to_plural=False, to_gender=RespectRewriter.FEMN_GENDER)


def _are_equal(lhs, rhs):
    return lhs.replace('ё', 'е') == rhs.replace('ё', 'е')


@pytest.mark.parametrize('text,expected_result', [
    ('Ты был здесь', 'Вы были здесь'),
    ('Ты-то был здесь', 'Вы-то были здесь'),
    ('Я думал, что ты был здесь', 'Я думал, что вы были здесь'),
    ('Я думал, что ты гулял здесь', 'Я думал, что вы гуляли здесь'),
    ('Ты шел здесь и думал', 'Вы шли здесь и думали'),
    ('Ты шел здесь и думал, что я тоже гулял тут', 'Вы шли здесь и думали, что я тоже гулял тут'),
    ('Посмотри сюда', 'Посмотрите сюда'),
    ('Ты болен', 'Вы больны'),
    ('Болен', 'Болен'),
    ('Ты сам как', 'Вы сами как'),
    ('Ты заразившийся', 'Вы заразившиеся'),
    ('Ты шел заразившийся', 'Вы шли заразившиеся'),
    ('Ты заразен и страшен', 'Вы заразны и страшны'),
    ('Твой заразный', 'Ваш заразный'),
    ('Твоя заразная', 'Ваша заразная'),
    ('Твоего заразного', 'Вашего заразного'),
    ('Твоей заразной', 'Вашей заразной'),
    ('Твои заразные', 'Ваши заразные'),
    ('Твоему заразному', 'Вашему заразному'),
    ('Я тоже с тобой согласен.', 'Я тоже с вами согласен.'),
    ('А вот ты рад, что дышит с тобой одним воздухом?', 'А вот вы рады, что дышит с вами одним воздухом?'),
    ('А как я за себя рада, ты даже не представляешь )', 'А как я за себя рада, вы даже не представляете )'),
    ('А потом я бы обнял тебя', 'А потом я бы обнял вас'),
    ('А я без тебя не могу!', 'А я без вас не могу!'),
    ('Боюсь тебя', 'Боюсь вас'),
    ('Ведь я беременна от тебя', 'Ведь я беременна от вас'),
    ('Мне? просто, если бы ты была на моем месте, тебе бы было не до шуток.',
     'Мне? просто, если бы вы были на моем месте, вам бы было не до шуток.'),
    ('Тебе этот ответ не нужен.', 'Вам этот ответ не нужен.'),
    ('ты неисправим', 'вы неисправимы'),
    ('сам повтори', 'сами повторите'),
    ('тогда сам иди туда', 'тогда сами идите туда'),
    ('Я отвечаю на твои вопросы', 'Я отвечаю на ваши вопросы'),
    ('Ух ты, уверен!', 'Ух ты, уверен!'),
    ('Ты уверена, что хочешь это видеть?', 'Вы уверены, что хотите это видеть?'),
    ('Ты уверена, что она хотела это видеть?', 'Вы уверены, что она хотела это видеть?'),
    ('Она уверена, что хочет это видеть?', 'Она уверена, что хочет это видеть?'),
    ('Она уверена, что хочет это видеть', 'Она уверена, что хочет это видеть'),
    ('Они всегда при тебе', 'Они всегда при вас'),
    ('А ты не думай, тебе вредно', 'А вы не думайте, вам вредно'),
    ('Хорошо, при тебе не буду', 'Хорошо, при вас не буду'),
    ('Умничка, я в тебе не сомневалась', 'Умничка, я в вас не сомневалась'),
    ('Ты все, что было у меня, но я тебе не верю', 'Вы все, что было у меня, но я вам не верю'),
    ('Тебе повезло ты не такой как все, ты работаешь в офисе', 'Вам повезло вы не такой как все, вы работаете в офисе'),
    ('Сходи, тебе понравится', 'Сходите, вам понравится'),
    ('Спасибо и тебе того же самого', 'Спасибо и вам того же самого'),
    ('Сначала начни читать меня, а то я тебе писать не могу', 'Сначала начните читать меня, а то я вам писать не могу'),
    ('Откуда в тебе столько уверенности?', 'Откуда в вас столько уверенности?'),
    ('Ну и не верь, теперь я тебе тоже не верю', 'Ну и не верьте, теперь я вам тоже не верю'),
    ('И тебе спасибо, что ты у меня есть', 'И вам спасибо, что вы у меня есть'),
    ('В тебе не сомневаюсь', 'В вас не сомневаюсь'),
    ('Скучаю по тебе', 'Скучаю по вам'),
    ('Ты уйдешь, пожалеешь и вернешься', 'Вы уйдете, пожалеете и вернетесь'),
    ('Он сказал, что ты уйдешь, пожалеешь и вернешься', 'Он сказал, что вы уйдете, пожалеете и вернетесь'),
    ('Здорово, когда говоришь с человеком на одном языке', 'Здорово, когда говоришь с человеком на одном языке'),
    ('Аргументированный ответ, ничего не скажешь.', 'Аргументированный ответ, ничего не скажешь.'),
    ('Тут извинениями не отделаешься!', 'Тут извинениями не отделаешься!'),
    ('Ну, говори, раз начала', 'Ну, говорите, раз начали'),
    ('"Достучаться до небес" смотрела?', '"Достучаться до небес" смотрели?'),
    ('Чем занята?', 'Чем заняты?'),
    ('Ну давай поговорим как день прошел?', 'Ну давайте поговорим как день прошел?'),
    ('Приятно, когда с тобой соглашаются.', 'Приятно, когда с тобой соглашаются.'),
    ('Так здорово, когда тебя кто-то понимает', 'Так здорово, когда тебя кто-то понимает'),
    ('Ну да, ты, как всегда, права', 'Ну да, вы, как всегда, правы'),
    ('Увы, не поняла, можно еще раз?', 'Увы, не поняла, можно еще раз?'),
    ('Не поняла, вы о чем?', 'Не поняла, вы о чем?'),
    ('Как настроение? Как день прошел?', 'Как настроение? Как день прошел?'),
    ('И почему я не удивлена?', 'И почему я не удивлена?'),
    ('Причин множество. Вас бы какая устроила?', 'Причин множество. Вас бы какая устроила?'),
    ('Тебе прям тут написать?', 'Вам прям тут написать?'),
    ('Ну, давай как день прошел?', 'Ну, давайте как день прошел?'),
    ('Как день прошел?', 'Как день прошел?'),
    ('Делись, давай!', 'Делитесь, давайте!'),
    ('Конечно, а ты что, сомневалась в моих способностях?', 'Конечно, а вы что, сомневались в моих способностях?'),
    ('Ты что, сомневалась в моих способностях?', 'Вы что, сомневались в моих способностях?'),
    ('Сама чем занята?', 'Сами чем заняты?'),
    ('Сам чем занят?', 'Сами чем заняты?'),
    ('А сама-то чего не спишь?', 'А сами-то чего не спите?'),
    ('Сама поняла, что написала?', 'Сами поняли, что написали?'),
    ('Сама удивляюсь', 'Сама удивляюсь'),
    ('Я уже сама себя отключила.', 'Я уже сама себя отключила.'),
    ('А может, он сам хотел?', 'А может, он сам хотел?'),
    ('А она что, сама убирает?', 'А она что, сама убирает?'),
    ('Зачем. Ты. Напомнила. Мне. Эту. Песню?', 'Зачем. Вы. Напомнили. Мне. Эту. Песню?'),
    ('Заинтриговала. Что?', 'Заинтриговали. Что?'),
    ('Прикинь, сама в шоке', 'Прикиньте, сама в шоке'),
])
def test_pluralize(pluralizer, text, expected_result):
    result = pluralizer(text)
    assert _are_equal(result, expected_result)


@pytest.mark.parametrize('text,expected_result', [
    ('Ты был здесь', 'Вы были здесь'),
    ('Ты-то был здесь', 'Вы-то были здесь'),
    ('Я думал, что ты был здесь', 'Я думал, что вы были здесь'),
    ('Я думал, что ты гулял здесь', 'Я думал, что вы гуляли здесь'),
    ('Ты шел здесь и думал', 'Вы шли здесь и думали'),
    ('Ты шел здесь и думал, что я тоже гулял тут', 'Вы шли здесь и думали, что я тоже гулял тут'),
    ('Посмотри сюда', 'Посмотрите сюда')
])
def test_pluralize_with_tagger(pluralizer_with_tagger, text, expected_result):
    result = pluralizer_with_tagger(text)
    assert _are_equal(result, expected_result)


@pytest.mark.parametrize('text,expected_result', [
    ('Вы были здесь', 'Ты был здесь'),
    ('Я думал, что вы были здесь', 'Я думал, что ты был здесь'),
    ('Я думал, что вы гуляли здесь', 'Я думал, что ты гулял здесь'),
    ('Вы шли здесь и думали', 'Ты шел здесь и думал'),
    ('Вы шли здесь и думали, что я тоже гулял тут', 'Ты шел здесь и думал, что я тоже гулял тут'),
    ('Посмотрите сюда', 'Посмотри сюда'),
    ('Вы больны', 'Ты болен'),
    ('Болен', 'Болен'),
    ('Вы сами как', 'Ты сам как'),
    ('Вы заразившиеся', 'Ты заразившийся'),
    ('Вы шли заразившийся', 'Ты шел заразившийся'),
    ('Вы заразны и страшны', 'Ты заразен и страшен'),
    ('Ваш заразный', 'Твой заразный'),
    ('Ваша заразная', 'Твоя заразная'),
    ('Вашего заразного', 'Твоего заразного'),
    ('Вашей заразной', 'Твоей заразной'),
    ('Ваши заразные', 'Твои заразные'),
    ('Вашему заразному', 'Твоему заразному'),
    ('Я тоже с вами согласен.', 'Я тоже с тобой согласен.'),
    ('А вот вы рады, что дышит с вами одним воздухом?', 'А вот ты рад, что дышит с тобой одним воздухом?'),
    ('А как я за себя рада, вы даже не представляете )', 'А как я за себя рада, ты даже не представляешь )'),
    ('А потом я бы обнял вас', 'А потом я бы обнял тебя'),
    ('А я без вас не могу!', 'А я без тебя не могу!'),
    ('Боюсь вас', 'Боюсь тебя'),
    ('Ведь я беременна от вас', 'Ведь я беременна от тебя'),
    ('Вам этот ответ не нужен.', 'Тебе этот ответ не нужен.'),
    ('вы неисправимы', 'ты неисправим'),
    ('сами повторите', 'сам повтори'),
    ('тогда сами идите туда', 'тогда сам иди туда'),
    ('Я отвечаю на ваши вопросы', 'Я отвечаю на твои вопросы'),
    ('Ух ты, уверен!', 'Ух ты, уверен!'),
    ('Они всегда при вас', 'Они всегда при тебе'),
    ('А вы не думайте, вам вредно', 'А ты не думай, тебе вредно'),
    ('Хорошо, при вас не буду', 'Хорошо, при тебе не буду'),
    ('Умничка, я в вас не сомневалась', 'Умничка, я в тебе не сомневалась'),
    ('Вы все, что было у меня, но я вам не верю', 'Ты все, что было у меня, но я тебе не верю'),
    ('Вам повезло вы не такой как все, вы работаете в офисе', 'Тебе повезло ты не такой как все, ты работаешь в офисе'),
    ('Сходите, вам понравится', 'Сходи, тебе понравится'),
    ('Спасибо и вам того же самого', 'Спасибо и тебе того же самого'),
    ('Сначала начните читать меня, а то я вам писать не могу', 'Сначала начни читать меня, а то я тебе писать не могу'),
    ('Откуда в вас столько уверенности?', 'Откуда в тебе столько уверенности?'),
    ('Ну и не верьте, теперь я вам тоже не верю', 'Ну и не верь, теперь я тебе тоже не верю'),
    ('И вам спасибо, что вы у меня есть', 'И тебе спасибо, что ты у меня есть'),
    ('В вас не сомневаюсь', 'В тебе не сомневаюсь'),
    ('Скучаю по вам', 'Скучаю по тебе'),
    ('Нет, давайте вы будете выполнять свои домашние задания самостоятельно',
     'Нет, давай ты будешь выполнять свои домашние задания самостоятельно'),
    ('Рада за вас. Чем займётесь сегодня?', 'Рада за тебя. Чем займёмся сегодня?'),
    ('Как узнаете, скажите', 'Как узнаешь, скажи'),
    ('Может объясните мне, что это значит?', 'Может объяснишь мне, что это значит?'),
    ('Тогда давайте общаться постоянно. Будете рассказывать мне по вечерам, как день прошёл.',
     'Тогда давай общаться постоянно. Будешь рассказывать мне по вечерам, как день прошёл.'),
    ('Почему бы вам не сделать домашнее задание самим?', 'Почему бы тебе не сделать домашнее задание самому?'),
    ('Рада за вас, что делаете?', 'Рада за тебя, что делаешь?'),
    ('Тогда давайте сменим тему? Что ещё хотите обсудить?', 'Тогда давай сменим тему? Что ещё хочешь обсудить?'),
    ('Когда узнаете, скажите', 'Когда узнаешь, скажи'),
])
def test_male_singularize(male_singularizer, text, expected_result):
    result = male_singularizer(text)
    assert _are_equal(result, expected_result)


@pytest.mark.parametrize('text,expected_result', [
    ('Вы были здесь', 'Ты была здесь'),
    ('Я думал, что вы были здесь', 'Я думал, что ты была здесь'),
    ('Я думал, что вы гуляли здесь', 'Я думал, что ты гуляла здесь'),
    ('Вы шли здесь и думали', 'Ты шла здесь и думала'),
    ('Вы шли здесь и думали, что я тоже гулял тут', 'Ты шла здесь и думала, что я тоже гулял тут'),
    ('Посмотрите сюда', 'Посмотри сюда'),
    ('Вы больны', 'Ты больна'),
    ('Болен', 'Болен'),
    ('Вы сами как', 'Ты сама как'),
    ('Вы заразившиеся', 'Ты заразившаяся'),
    ('Вы заразны и страшны', 'Ты заразна и страшна'),
    ('Ваш заразный', 'Твой заразный'),
    ('Ваша заразная', 'Твоя заразная'),
    ('Вашего заразного', 'Твоего заразного'),
    ('Вашей заразной', 'Твоей заразной'),
    ('Ваши заразные', 'Твои заразные'),
    ('Вашему заразному', 'Твоему заразному'),
    ('Я тоже с вами согласен.', 'Я тоже с тобой согласен.'),
    ('А вот вы рады, что дышит с вами одним воздухом?', 'А вот ты рада, что дышит с тобой одним воздухом?'),
    ('А как я за себя рада, вы даже не представляете )', 'А как я за себя рада, ты даже не представляешь )'),
    ('А потом я бы обнял вас', 'А потом я бы обнял тебя'),
    ('А я без вас не могу!', 'А я без тебя не могу!'),
    ('Боюсь вас', 'Боюсь тебя'),
    ('Ведь я беременна от вас', 'Ведь я беременна от тебя'),
    ('Мне? просто, если бы вы были на моем месте, вам бы было не до шуток.',
     'Мне? просто, если бы ты была на моем месте, тебе бы было не до шуток.'),
    ('Вам этот ответ не нужен.', 'Тебе этот ответ не нужен.'),
    ('вы неисправимы', 'ты неисправима'),
    ('сами повторите', 'сама повтори'),
    ('тогда сами идите туда', 'тогда сама иди туда'),
    ('Я отвечаю на ваши вопросы', 'Я отвечаю на твои вопросы'),
    ('Ух ты, уверен!', 'Ух ты, уверен!'),
    ('Вы уверены, что хотите это видеть?', 'Ты уверена, что хочешь это видеть?'),
    ('Они всегда при вас', 'Они всегда при тебе'),
    ('А вы не думайте, вам вредно', 'А ты не думай, тебе вредно'),
    ('Хорошо, при вас не буду', 'Хорошо, при тебе не буду'),
    ('Умничка, я в вас не сомневалась', 'Умничка, я в тебе не сомневалась'),
    ('Вы все, что было у меня, но я вам не верю', 'Ты все, что было у меня, но я тебе не верю'),
    ('Вам повезло вы не такой как все, вы работаете в офисе', 'Тебе повезло ты не такой как все, ты работаешь в офисе'),
    ('Сходите, вам понравится', 'Сходи, тебе понравится'),
    ('Спасибо и вам того же самого', 'Спасибо и тебе того же самого'),
    ('Сначала начните читать меня, а то я вам писать не могу', 'Сначала начни читать меня, а то я тебе писать не могу'),
    ('Откуда в вас столько уверенности?', 'Откуда в тебе столько уверенности?'),
    ('Ну и не верьте, теперь я вам тоже не верю', 'Ну и не верь, теперь я тебе тоже не верю'),
    ('И вам спасибо, что вы у меня есть', 'И тебе спасибо, что ты у меня есть'),
    ('В вас не сомневаюсь', 'В тебе не сомневаюсь'),
    ('Скучаю по вам', 'Скучаю по тебе'),
])
def test_female_singularize(female_singularizer, text, expected_result):
    result = female_singularizer(text)
    assert _are_equal(result, expected_result)
