package ru.yandex.quasar.billing.dao;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.Getter;
import lombok.ToString;
import lombok.With;
import org.springframework.data.annotation.Id;
import org.springframework.data.relational.core.mapping.Column;

import ru.yandex.quasar.billing.beans.DeliveryInfo;
import ru.yandex.quasar.billing.beans.PricingOption;

@Getter
@ToString
@AllArgsConstructor(access = AccessLevel.PACKAGE)
@Builder()
public class PurchaseOffer {
    @Id
    @With
    @Column("purchase_offer_id")
    private final Long id;

    /**
     * may be null if the purchase offer is created for anonymous user.
     * Has to be bind to a user before any further processing
     */
    @Nullable
    private final String uid;
    @Nonnull
    private final Long skillInfoId;
    /**
     * UUID of the purchase offer
     */
    @Nonnull
    private final String uuid;
    /**
     * external purchase request ID generated by the caller
     */
    @Nonnull
    private final String purchaseRequestId;

    @Nonnull
    private final String title;

    @Nonnull
    private final PurchaseOfferOptions pricingOptions;

    @Nullable
    private final String imageUrl;
    @Nullable
    private final String description;
    @Nonnull
    private final String skillCallbackUrl;
    @Nonnull
    private final String skillSessionId;
    @Nonnull
    private final String skillUserId;
    private final String skillName;
    private final String skillImageUrl;

    @Column("delivery_info")
    @Nullable
    private final DeliveryInfo deliveryInfo;

    @Nonnull
    private final Instant createdAt;
    @Nullable
    @Column("initial_device_id")
    private final String deviceId;
    @Nonnull
    private PurchaseOfferStatus status;
    @Column("merchant_key")
    @Nonnull
    private final String merchantKey;

    @Column("test_payment")
    private final boolean testPayment;

    @Nullable
    @Column("webhook_request")
    private final String webhookRequest;

    public static PurchaseOfferBuilder builder(
            @Nullable String uid,
            Long skillId,
            String purchaseRequestId,
            List<PricingOption> pricingOptions,
            String skillCallbackUrl
    ) {
        return new PurchaseOfferBuilder()
                .uid(uid)
                .status(PurchaseOfferStatus.NEW)
                .skillInfoId(skillId)
                .purchaseRequestId(purchaseRequestId)
                .pricingOptions(new PurchaseOfferOptions(pricingOptions))
                .uuid(UUID.randomUUID().toString())
                .createdAt(Instant.now())
                .skillCallbackUrl(skillCallbackUrl);
    }

    @Nonnull
    public List<PricingOption> getPricingOptions() {
        return pricingOptions.getOptions();
    }

    public Optional<DeliveryInfo> getDeliveryInfo() {
        return Optional.ofNullable(deliveryInfo);
    }

    public void setStatus(@Nonnull PurchaseOfferStatus status) {
        this.status = status;
    }

    @Data
    static class PurchaseOfferOptions {
        private final List<PricingOption> options;
    }
}
