import logging

import alice.tests.library.uniclient as uniclient
import alice.tests.library.url as url

from alice.tests.library.locale.tanker_localizer import TankerLocalizer

from . import opus


class _AliceWrapperBase(object):
    def __init__(self, alice):
        self._alice = alice

    def __call__(self, command):
        return self._alice(command)

    def __getattr__(self, attr):
        orig_attr = getattr(self._alice, attr)
        if callable(orig_attr):
            def _wrap(*args, **kwargs):
                return orig_attr(*args, **kwargs)
            return _wrap
        return orig_attr


class AliceWrapper(_AliceWrapperBase):
    def __init__(self, alice, backend_version):
        super().__init__(alice)
        self._backend_version = backend_version
        self._request_time = [None, None]

    def __call__(self, command):
        self._request_time[0] = self._alice.datetime_now
        response = self._alice(command)
        self._request_time[1] = self._alice.datetime_now
        return response

    @property
    def version(self):
        return self.backend_version

    @property
    def request_time(self):
        return self._request_time


class AliceVoiceWrapper(_AliceWrapperBase):
    def __init__(self, alice, voice_marker, regen_opus=False):
        super().__init__(alice)
        self._voice_tags = voice_marker.args
        self._regen_opus = regen_opus
        self._uniclient = None

    def __call__(self, command, voice=None):
        voice_tags = voice or self._voice_tags
        voice_command = opus.load(command, voice_tags)
        if not voice_command or self._regen_opus:
            assert not voice_tags, ('Voice with special tags will not be autogenerated. '
                                    'You have to upload them to YDB yourself.')
            logging.info(f'Generate opus for text: {command}')
            voice_command = self._generate_voice(command)
            opus.save(command, voice_tags, voice_command)
        return super().__call__(voice_command)

    def _generate_voice(self, text):
        if not self._uniclient:
            settings = uniclient.AliceSettings(
                log_type='cerr',
                uniproxy_url=url.Uniproxy.prod,
                auth_token='069b6659-984b-4c5f-880e-aaedcfd84102',
            )
            application = uniclient.Application(
                app_id='ru.yandex.evo.tests.generate.opus',
                uuid=self._alice._application.Uuid,
                lang=self._alice._application.Lang,
            )
            self._uniclient = uniclient.Uniclient(settings, application)
        return self._uniclient.generate_voice(text)


class TankerLocalizerAliceWrapper(_AliceWrapperBase):
    def __init__(self, alice, tanker_data, locale_name, test_group_id):
        super().__init__(alice)
        self._tanker_localizer = TankerLocalizer(tanker_data, locale_name, test_group_id)

    def __call__(self, command):
        localized_command = self._tanker_localizer.localize_command(command)
        return super().__call__(localized_command)
