__author__ = 'lpshka'
__version__ = '1.004'

import sys
import time
import json
import requests
from datetime import datetime, timedelta

import logging
from argparse import ArgumentParser, Action

_logger = logging.getLogger(__name__)

GIGAMINER_PARSE_TRAFFIC_SESSIONS = "https://gigaminer-ng.qe.yandex-team.ru/api/miner/start-traffic"
GIGAMINER_GET_STATUS = "https://gigaminer-ng.qe.yandex-team.ru/api/miner/tasks/%s"
GIGAMINER_GET_INFO = "https://gigaminer-ng.qe.yandex-team.ru/api/miner/tasks/%s/info"
GIGAMINER_GET_RESULT = "https://gigaminer-ng.qe.yandex-team.ru/api/miner/tasks/%s/queries"

HEADERS = {"Content-Type": "application/json; charset=UTF-8"}


class PostRequest(object):
    @staticmethod
    def request(request_function, address, headers, data=None, params=None):
        _logger.info("Address: %s." % address)
        _logger.info("Headers: %s." % headers)
        _logger.info("Data: %s." % data)
        iteration = 0
        while True:
            try:
                response = request_function(url=address, data=json.dumps(data), headers=headers, params=params)
                try:
                    response_json = response.json()
                except ValueError:
                    response_json = None
                if isinstance(response_json, dict):
                    _logger.info('Response: %s' % response_json)
                    return response_json
                elif response.ok:
                    return response.text
            except Exception as e:
                _logger.error(str(e))
                if iteration == 50:
                    _logger.error("Program exits with request error.")
                    raise
            iteration += 1
            time.sleep(300)


class ToUpperKey(Action):
    def __init__(self, option_strings, dest, nargs=None, **kwargs):
        if nargs is not None:
            raise ValueError("nargs not allowed")
        super(ToUpperKey, self).__init__(option_strings, dest, **kwargs)

    def __call__(self, parser, namespace, values, option_string=None):
        try:
            setattr(namespace, self.dest, str(values).upper())
        except Exception as e:
            _logger.warning(str(e))
            setattr(namespace, self.dest, None)


def main():
    argparser = ArgumentParser(description="Queries miner.",
                               add_help=True, usage="""%(prog)s [OPTIONS]
                               """)

    argparser.add_argument("-O", "--owner", dest="owner", default="lpshka", help="owner")
    argparser.add_argument("-d", "--description", dest="description", nargs='+', default="test", help="description")
    argparser.add_argument('-e', "--end-date", dest="end_date",
                           default=(datetime.now() - timedelta(days=3)).strftime("%Y-%m-%d"), help="last date.")
    argparser.add_argument('-C', "--count-of-days", dest="count_of_days", type=int, default=7,
                           help="count of days for mining")
    argparser.add_argument('-c', "--count", dest="count", default=1000, type=int, help="count of queries")
    argparser.add_argument("--source", dest="source", default="SIMILAR_GROUP", help="source, default: SIMILAR_GROUP")
    argparser.add_argument("--search-engine", dest="search_engine", default="GOOGLE",
                           help="search engine, default: GOOGLE")
    argparser.add_argument("--dom-region", dest="dom_region", action=ToUpperKey, default=None,
                           help="dom region, default: id")
    argparser.add_argument("--device-filter", dest="device_filter", default="DESKTOP",
                           help="device filter, default: DESKTOP")
    argparser.add_argument("--service-type", dest="service_type", default="WEB",
                           help="service type, default: WEB")

    argparser.add_argument("-o", "--output-queries", dest="output_queries", default="queries.tsv",
                           help="output queries tsv file")

    argparser.add_argument("-q", "--add-qids", dest="add_qids", default=False, action="store_true",
                           help="add qids")

    argparser.add_argument("-v", "--verbose", dest="verbose", default=False, action="store_true",
                           help="output debug information")

    opts = argparser.parse_args()
    loglevel = logging.INFO
    if opts.verbose:
        loglevel = logging.DEBUG
    logging.basicConfig(level=loglevel,
                        format="%(asctime)s - %(levelname)s - %(message)s",
                        stream=sys.stdout)

    _logger.info(__author__)
    _logger.info(__version__)

    _logger.info("Begin mine queries.")
    _logger.info("Input owner: %s.", opts.owner)
    _logger.info("Input description: %s.", ' '.join(opts.description))
    if '-' in opts.end_date:
        begin_date =\
            (datetime.strptime(opts.end_date, "%Y-%m-%d").date() - timedelta(days=opts.count_of_days)).strftime("%Y-%m-%d")
    else:
        begin_date =\
            (datetime.strptime(opts.end_date, "%Y%m%d").date() - timedelta(days=opts.count_of_days)).strftime("%Y-%m-%d")
        end_date = (datetime.strptime(opts.end_date, "%Y%m%d").date()).strftime("%Y-%m-%d")
    _logger.info("Input begin date: %s.", begin_date)
    _logger.info("Input end date: %s.", end_date)
    _logger.info("Input count of days: %d.", opts.count_of_days)

    _logger.info("Input count: %d.", opts.count)
    _logger.info("Input source: %s.", opts.source)
    _logger.info("Input search engine: %s.", opts.search_engine)
    _logger.info("Input dom region: %s.", opts.dom_region)
    _logger.info("Input device filter: %s.", opts.device_filter)
    _logger.info("Input service type: %s.", opts.service_type)
    _logger.info("Input add qids: %s.", str(opts.add_qids))
    _logger.info("Output queries file: %s.", opts.output_queries)

    request_data = {
        "owner": opts.owner,
        "description": ' '.join(opts.description),
        "sourceTables": {
            "source": opts.source,
            "startDate": begin_date,
            "endDate": end_date},
        "numQueries": opts.count,
        "searchEngine": opts.search_engine,
        "serviceType": opts.service_type,
        "domRegion": opts.dom_region,
        "deviceFilter": opts.device_filter
    }

    response = PostRequest.request(requests.post, GIGAMINER_PARSE_TRAFFIC_SESSIONS, HEADERS, request_data, None)
    task_id = response.get("id")
    status = response.get("status")
    while status != "FINISHED":
        time.sleep(600)
        response = PostRequest.request(requests.get, GIGAMINER_GET_STATUS % task_id, HEADERS, None, None)
        _logger.info("Status: %s." % response.get("status"))
        status = response.get("status")
        if status == "FAILED":
            raise Exception(response["statusMessage"])

    response = PostRequest.request(requests.get, GIGAMINER_GET_INFO % task_id, HEADERS, None, None)
    _logger.info("Info: %s." % response)
    result = PostRequest.request(requests.get, GIGAMINER_GET_RESULT % task_id, HEADERS, None, None)

    qid = 1
    with open(opts.output_queries, 'wb') as output_queries:
        for line in result.strip().split('\n'):
            fields = line.strip().split('\t')
            try:
                if opts.add_qids:
                    output_queries.write("%s\t%s\t%s\n" % (str(qid), fields[2], fields[3]))
                else:
                    output_queries.write("%s\t%s\n" % (fields[2], fields[3]))
                qid += 1
            except UnicodeEncodeError:
                continue

    _logger.info("End mine queries.")


if __name__ == '__main__':
    start_time = time.time()
    main()
    _logger.info("--- time of program execution: %f seconds ---" % (time.time() - start_time))
