package com.ajkon.Kernel;

import com.ajkon.Model.Cell;
import com.ajkon.Model.ValueSelectMethod;
import javafx.util.Pair;

import java.util.Date;
import java.util.List;

import static java.util.stream.Collectors.toList;


/**
 * Класс выполняющий расчёт метрик по данным переданным в ячейках.
 */
public class Calc {


    private final List<List<Cell>> cells;


    public Calc(List<List<Cell>> cells) {
        this.cells = cells;
    }

    public void calcMetrics() {
        cells.stream()
                .flatMap(i -> i.stream())
                .filter(j -> (!j.isWorldMetrics || !j.recalcWorld))
                .forEach(k -> calcMetrics(k));
        cells.stream()
                .forEach(k -> recalcWorld(k));
    }


    private static final void calcMetrics(Cell cell) {
        Double yFirsValue = getValue(cell.yData, cell.firstStart, cell.firstEnd, cell.valueSelectMethod);
        Double ySecondValue = getValue(cell.yData, cell.secondStart, cell.secondEnd, cell.valueSelectMethod);
        Double gFirsValue = getValue(cell.gData, cell.firstStart, cell.firstEnd, cell.valueSelectMethod);
        Double gSecondValue = getValue(cell.gData, cell.secondStart, cell.secondEnd, cell.valueSelectMethod);
        if (yFirsValue != null && ySecondValue != null) {
            switch (cell.calcMethod) {
                case DeltaAbsoluteMilli:
                    cell.result = (ySecondValue - yFirsValue) * 1000.0;
                    break;
                case DeltaPercentage:
                    cell.result = ((ySecondValue - yFirsValue) / yFirsValue) * 100.0;
                    break;
                case YandexVsGoogle:
                    if (gFirsValue != null && gSecondValue != null) {
                        cell.result = ((ySecondValue / gSecondValue) - (yFirsValue / gFirsValue)) * 100;
                    } else {
                        cell.result = Double.NaN;
                    }
                    break;
                default:
                    cell.result = ySecondValue - yFirsValue;
                    break;
            }
        } else {
            cell.result = Double.NaN;
        }
        if (cell.result != null && !cell.result.isNaN()) {
            cell.worldresult = cell.result * cell.countryPriority;
        } else {
            cell.worldresult = Double.NaN;
        }
    }


    private static final void recalcWorld(List<Cell> rowCell) {
        Cell worldCell = rowCell.stream()
                .filter(l -> (l.isWorldMetrics && l.recalcWorld))
                .findAny().orElse(null);
        if (worldCell == null) return;
        worldCell.isMetrics = true;
        List<Double> countryWorldResults = rowCell.stream()
                .filter(l -> !l.isWorldMetrics && l.worldresult != null && !l.worldresult.isNaN())
                .map(l -> l.worldresult)
                .collect(toList());
        if (countryWorldResults.size() == 0) return;
        worldCell.result = countryWorldResults.stream()
                .mapToDouble(Double::doubleValue).sum();
        worldCell.worldresult = worldCell.result;
    }


    private static final Double getValue(List<Pair<Date, Double>> data, Date start, Date end, ValueSelectMethod selectMethod) {
        if (data == null || data.size() == 0)
            return null;
        List<Double> values = data.stream()
                .filter(v -> v.getKey().getTime() > start.getTime()
                        && v.getKey().getTime() < end.getTime())
                .map(l -> l.getValue())
                .filter(l -> (l != null && !l.isNaN() && !l.isInfinite()))
                .collect(toList());
        if (values.size() == 0) return null;
        double value;
        switch (selectMethod) {
            case Maximum:
                value = values.stream().mapToDouble(l -> l).max().getAsDouble();
                return value;
            default:
                List<Double> sortedValues = values.stream().mapToDouble(l -> l).sorted().boxed().collect(toList());
                int index = (sortedValues.size() - 1) / 2;
                boolean isEvenNumber = (sortedValues.size() % 2 == 0);
                if (isEvenNumber) {
                    value = (sortedValues.get(index).doubleValue() + sortedValues.get(index + 1).doubleValue()) / 2;
                    return value;
                } else {
                    value = sortedValues.get(index).doubleValue();
                    return value;
                }
        }
    }
}
