#!/usr/bin/env python
# -*- coding: utf-8 -*-
from nile.api.v1 import (
    cli,
    Record,
    aggregators as na,
    extractors as ne
)
from qb2.api.v1 import filters as sf
from datetime import datetime


def get_visits(job, start_date, end_date):
    return job.table(
        '//home/zapravki/production/stations_stat/visits_uuids/{%s..%s}' % (start_date, end_date)
    ).project(
        ne.all(),
        event_date=ne.custom(
            lambda date_time: datetime.strftime(
                datetime.strptime(date_time, '%Y%m%dT%H%M%S'), '%Y-%m-%d')).with_type(str),
        event_datetime=ne.custom(
            lambda date_time: datetime.strftime(
                datetime.strptime(date_time, '%Y%m%dT%H%M%S'), '%Y-%m-%d %H:%M:%S')).with_type(str),
        station_id='group'
    )


def get_navi_users(job, start_date, end_date):
    return job.table('//statbox/heavy-dict/mobile_device_ids/{%s..%s}' % (start_date, end_date)
        ).filter(
            sf.equals('api_key', 30488)
        ).project(
            'device_id',
            'uuid',
            'app_version',
            event_date=ne.custom(lambda event_datetime: event_datetime[:10]).with_type(str),
            puid=ne.custom(lambda puid: puid if puid is not None else '').with_type(str)
        ).groupby(
            'device_id',
            'event_date',
            'uuid'
        ).aggregate(
            puid=na.max('puid'),
            app_version=na.max('app_version')
    )


def prepare_taxi_devices(job, start_date, end_date):
    return job.concat(
        job.table('//home/geo-analytics/androsovg/NAVIANALYTICS-300/taxi_drivers'),
        job.table('//statbox/metrika-mobile-log/{%s..%s}' % (start_date, end_date)
            ).qb2(
                log='metrika-mobile-log',
                fields=['device_id'],
                filters=[
                    sf.equals('api_key', 121179),
                    sf.defined('device_id')
                ]
            ).project(
            DeviceID='device_id'
            )
    ).unique(
            'DeviceID'
    ).put(
            '//home/geo-analytics/androsovg/NAVIANALYTICS-300/taxi_drivers'
    )


@cli.statinfra_job
def make_job(job, options):
    start_date, end_date = options.dates[0], options.dates[-1]
    job.concat(
        job.table('//home/zapravki/production/stations_stat/users-refuelling'),
        get_navi_users(
            job, start_date, end_date
        ).join(
            prepare_taxi_devices(job, start_date, end_date),
            by_left='device_id',
            by_right='DeviceID',
            type='left',
            assume_unique_right=True
        ).project(
            'device_id',
            'app_version',
            'event_date',
            'uuid',
            'DeviceID',
            'puid'
        ).join(
            get_visits(job, start_date, end_date),
            by=['uuid', 'event_date'],
            type='inner',
        ).project(
            'device_id',
            'uuid',
            'app_version',
            'event_datetime',
            'time_spent',
            'puid',
            station_id='group',
            is_taxi_driver=ne.custom(lambda DeviceID: DeviceID is not None).with_type(bool),
        )
    ).put('//home/zapravki/production/stations_stat/users-refuelling')

    return job


if __name__ == '__main__':
    cli.run()
