#!/usr/bin/env python
# -*- coding: utf-8 -*-
from nile.api.v1 import (
    cli,
    Record,
    clusters,
    aggregators as na,
    extractors as ne,
)
from qb2.api.v1 import filters as sf, extractors as se
import datetime


def get_visits(job, start_date, end_date):
    return job.table(
        '//home/zapravki/production/stations_stat/visits_uuids/{%s..%s}' % (start_date, end_date)
    ).project(
        ne.all(),
        event_date=ne.custom(
            lambda date_time: datetime.datetime.strftime(
                datetime.datetime.strptime(date_time, '%Y%m%dT%H%M%S'), '%Y-%m-%d')).with_type(str),
        event_datetime=ne.custom(
            lambda date_time: datetime.datetime.strftime(
                datetime.datetime.strptime(date_time, '%Y%m%dT%H%M%S'), '%Y-%m-%d %H:%M:%S')).with_type(str),
        station_id='group'
    )


def get_navi_users(job, start_date, end_date):
    return job.table('//statbox/metrika-mobile-log/{%s..%s}' % (start_date, end_date)
                    ).qb2(
        log='metrika-mobile-log',
        fields=[
            'device_id',
            'date',
            'uuid',
            'event_date',
            'app_platform',
            'app_version',
            'api_key',
            'puid',
            se.log_field('ADVID')
        ],
        filters=[
            sf.one_of('api_key', [30488, 4]),
            sf.defined('device_id'),
        ]
        ).groupby(
            'api_key',
            'app_platform',
            'device_id',
            'event_date',
            'uuid'
        ).aggregate(
            puid=na.max('puid'),
            app_version=na.max('app_version'),
            ADVID=na.max('ADVID'),
    )


def prepare_taxi_devices(job, start_date, end_date):
    return job.concat(
        job.table('//home/geo-analytics/androsovg/NAVIANALYTICS-390/taxi_drivers'),
        job.table('//statbox/metrika-mobile-log/{%s..%s}' % (start_date, end_date)
            ).qb2(
                log='metrika-mobile-log',
                fields=[se.log_field('ADVID')],
                filters=[
                    sf.equals('api_key', 121179),
                    sf.defined('ADVID')
                ])
    ).unique(
            'ADVID'
    ).put(
            '//home/geo-analytics/androsovg/NAVIANALYTICS-390/taxi_drivers'
    )

@cli.statinfra_job
def make_job(job, options):
    start_date, end_date = options.dates[0], options.dates[-1]
    job.concat(
        job.table('//home/zapravki/production/stations_stat/users-refuelling-v2'),
        get_navi_users(
            job, start_date, end_date
        ).join(
            prepare_taxi_devices(job, start_date, end_date).project(taxi_advid='ADVID'),
            by_left='ADVID',
            by_right='taxi_advid',
            type='left',
            assume_unique_right=True
        ).project(
            'api_key',
            'device_id',
            'app_version',
            'event_date',
            'uuid',
            'taxi_advid',
            'puid'
        ).join(
            get_visits(job, start_date, end_date),
            by=['uuid', 'event_date'],
            type='inner',
        ).project(
            'api_key',
            'device_id',
            'uuid',
            'app_version',
            'event_datetime',
            'time_spent',
            'clid',
            'puid',
            station_id='group',
            is_taxi_driver=ne.custom(lambda taxi_advid: taxi_advid is not None).with_type(bool)
        )
    ).put('//home/zapravki/production/stations_stat/users-refuelling-v2')

    return job


if __name__ == '__main__':
    cli.run()
