#!/usr/bin/env python
# -*- coding: utf-8 -*-
from nile.api.v1 import (
    cli,
    Record,
    aggregators as na,
    extractors as ne,
    statface as ns
)
from qb2.api.v1 import (
    filters as sf,
    extractors as se
)

TARGET_EVENTS = [
        'gas-stations.taximeter_open',
        'gas-station-card.opened',
        'gas-station-card.auto-closed',
        'gas-station-card.closed',
        'gas-stations.select_column',
        'gas-stations.order',
        'gas-stations.order_success',
        'gas-stations.order_error'
    ]


def get_index(list_, elem_, min_idx):
    try:
        return list_.index(elem_, min_idx)
    except ValueError:
        return -1


def get_value(event_value, key1, key2, val):
    try:
        return event_value[key1][key2] == val
    except (KeyError, ValueError, TypeError):
        return False


def get_conv(nxt, prv):
    return float(nxt) / prv if prv > 0 else 0


def reducer(groups):
    for key, records in groups:
        funnel = [
            {'level': 'show', 'pos': -1, 'event': ['taximeter.show']},
            {'level': 'click', 'pos': -1, 'event': ['taximeter.click']},
            {'level': 'open', 'pos': -1, 'event': ['gas-stations.taximeter_open']},
            {'level': 'card', 'pos': -1, 'event': ['gas-station-card.opened']},
            {'level': 'column', 'pos': -1, 'event': ['gas-stations.select_column']},
            {'level': 'button', 'pos': -1, 'event': ['gas-stations.order_success', 'gas-stations.order_error']},
            {'level': 'order', 'pos': -1, 'event': ['gas-stations.order_success']}
        ]
        events = []
        values = []
        source = 'Unknown'
        authorized = False
        puid = None
        pass_by = 0
        app_version = ''
        for record in records:
            if record['puid'] is not None and len(record['puid']) > 1:
                authorized = True
                puid = record['puid']
            if record['api_key_str'] in ('30488', 30488):
                app_version = record['app_version']
            if record['new_event_name'] == 'gas-station-card.closed':
                if record['event_value'].get('automatically'):
                    new_event_name = 'gas-station-card.auto-closed'
                else:
                    continue
            else:
                new_event_name = record['new_event_name']
            events.append({
                'event_name': new_event_name,
                'event_timestamp': record['event_timestamp'],
                'value': record['event_value'] if record['api_key_str'] in ('30488', 30488) else ''
            })
        ########
        event_list = map(lambda x: x['event_name'], events)

        if (len({
                   'taximeter.show',
                   'taximeter.click',
                   'gas-stations.taximeter_open'
               }.intersection(set(event_list))) == 0
               and 'gas-station-card.opened' in event_list):
            funnel = funnel[3:]
            source = 'Navi'
        else:
            source = 'Taximeter'
        prev = 0
        for num, level in enumerate(funnel):
            minlist = [get_index(event_list, event, prev) for event in level['event']]
            min_index = min(minlist) if -1 not in minlist else max(minlist)
            if min_index == -1:
                break
            prev = min_index if min_index > -1 else prev
            funnel[num]['pos'] = min_index
        result_dict = {i['level']: 1 if i['pos'] > -1 else 0 for i in funnel}
        auto_open = events[funnel[0]['pos']]['value']['automatically'] \
            if 'Navi' in source and 'automatically' in events[funnel[0]['pos']]['value'] \
            else 'Unknown'

        if source == 'Navi':
            auto_close_idx = get_index(
                event_list,
                'gas-station-card.auto-closed',
                funnel[0]['pos']
            )
            if auto_close_idx != -1:
                auto_close_time = events[auto_close_idx]['event_timestamp']

                if {
                    'gas-station-card.opened',
                    'gas-station-card.auto-closed'
                } == set(event_list) and \
                        0 <= events[funnel[1]['pos']]['event_timestamp'] - auto_close_time <= 20:
                    pass_by = 1

        if source != 'Unknown':
            yield Record(
                key,
                app_version=app_version,
                puid=puid,
                source=['_total_', source],
                authorized=['_total_', authorized],
                auto_open=['_total_', auto_open],
                show=result_dict.get('show', 0),
                click=result_dict.get('click', 0),
                open=result_dict.get('open', 0),
                hold_open=result_dict.get('card', 0) - pass_by,
                card=result_dict.get('card', 0),
                column=result_dict.get('column', 0),
                button=result_dict.get('button', 0),
                order=result_dict.get('order', 0)
            )


def get_report(statface_client, scale='d'):
    return ns.StatfaceReport() \
        .path('Mobile_Navig_Static/petrol_stations/petrol_conversion_v11') \
        .title('Конверсия в заправку v1.1') \
        .scale(scale) \
        .dimensions(
        ns.Date('fielddate'),
        ns.StringSelector('app_platform').title('Платформа'),
        ns.StringSelector('source').title('Источник'),
        ns.StringSelector('authorized').title('Авторизованный'),
        ns.StringSelector('has_cards').title('Привязана карта'),
        ns.StringSelector('auto_open').title('Способ открытия'),
        ns.StringSelector('has_orders').title('Новизна')
    ) \
        .measures(ns.Number('show').title('Показы в Таксометре'),
                  ns.Number('click').title('Клики в Таксометре'),
                  ns.Number('open').title('Открытия из Таксометра'),
                  ns.Number('card').title('Открытие карточки АЗС'),
                  ns.Number('hold_open').title('Заезд на АЗС'),
                  ns.Number('column').title('Выбор колонки'),
                  ns.Number('button').title('Нажатие на кнопку оплаты'),
                  ns.Number('order').title('Успешный заказ'),
                  ns.Percentage('conversion_click').title('Конверсия в клик в Таксометре'),
                  ns.Percentage('conversion_open').title('Конверсия в открытие из Таксометра'),
                  ns.Percentage('conversion_card').title('Конверсия в открытие карточки АЗС'),
                  ns.Percentage('conversion_hold_open').title('Конверсия в заезд на АЗС'),
                  ns.Percentage('conversion_column').title('Конверсия в выбор колонки'),
                  ns.Percentage('conversion_button').title('Конверсия в нажатие кнопки оплаты'),
                  ns.Percentage('conversion_order').title('Конверсия в успешный заказ'),
                  ns.Percentage('conversion_prev_click').title('Конверсия в клик в Таксометре (пред)'),
                  ns.Percentage('conversion_prev_open').title('Конверсия в открытие карточки из Таксометра (пред)'),
                  ns.Percentage('conversion_prev_card').title('Конверсия в открытие карточки АЗС (пред)'),
                  ns.Percentage('conversion_prev_hold_open').title('Конверсия в заезд на АЗС (пред)'),
                  ns.Percentage('conversion_prev_column').title('Конверсия в выборк колонки (пред)'),
                  ns.Percentage('conversion_prev_button').title('Конверсия в нажатие кнопки оплаты (пред)'),
                  ns.Percentage('conversion_prev_order').title('Конверсия в успешный заказ (пред)')) \
        .client(statface_client)


def get_previous_orders(job):
    return job\
        .table('home/zapravki/production/replica/mongo/struct/orders_full')\
        .project(
            date=ne.custom(lambda DateEnd: DateEnd[:10]),
            puid=ne.custom(lambda YandexUser:
                YandexUser['UserId'] if YandexUser is not None and 'UserId' in YandexUser else None)
        )\
        .filter(
            sf.defined('puid')
        )\
        .groupby('puid')\
        .aggregate(min_date=na.min('date'))


def get_taximeter_metrika(job, start_date, end_date):
    return job.table('statbox/metrika-mobile-log/{%s..%s}' % (start_date, end_date))\
        .qb2(
        log='metrika-mobile-log',
        fields=[
            'device_id',
            'date',
            'app_platform',
            'api_key_str',
            'event_timestamp',
            'event_name',
            'puid',
            se.mobile.event_value(),
            se.log_field('ADVID')
        ],
        filters=[
            sf.equals('api_key_str', '121179'),
            sf.defined('device_id'),
            sf.equals('event_name', 'ui_event'),
            sf.custom(lambda event_value:
                      get_value(event_value, 'ToggleValueParams', 'menu/benzin', 'TRUE') or
                      get_value(event_value, 'UiParams', 'uiComponentName', 'menu/benzin'))
        ]
    )\
        .project(
        'date',
        'app_platform',
        'api_key_str',
        'event_timestamp',
        'event_name',
        'puid',
        'event_value',
        device_id='ADVID',
        fielddate='date',
        new_event_name=ne.custom(
            lambda event_value:
            'taximeter.show' if get_value(event_value, 'ToggleValueParams', 'menu/benzin',
                                          'TRUE') else 'taximeter.click')
    )


def get_navi_metrika(job, start_date, end_date):
    return job.table('statbox/metrika-mobile-log/{%s..%s}' % (start_date, end_date)) \
        .qb2(
        log='metrika-mobile-log',
        fields=[
            'device_id',
            'date',
            'app_platform',
            'app_version',
            'api_key_str',
            'event_timestamp',
            'event_name',
            'puid',
            se.mobile.event_value(),
            se.log_field('ADVID')
        ],
        filters=[
            sf.equals('api_key_str', '30488'),
            sf.defined('device_id'),
            sf.one_of('event_name', TARGET_EVENTS)
        ]) \
        .project(
        'date',
        'app_platform',
        'app_version',
        'api_key_str',
        'event_timestamp',
        'event_name',
        'puid',
        'event_value',
        device_id=ne.custom(
            lambda ADVID, device_id, app_platform: ADVID if app_platform == 'Android' else device_id),
        fielddate='date',
        new_event_name='event_name'
    ) \
        .filter(
            sf.custom(
                lambda app_platform, app_version:
                app_version >= '3.73' if app_platform == 'Android' else app_version >= '374')
    )


@cli.statinfra_job
def make_job(job, options, statface_client):
    start_date, end_date = options.dates[0], options.dates[-1]
    (job
        .concat(
            get_taximeter_metrika(job, start_date, end_date),
            get_navi_metrika(job, start_date, end_date))
        .groupby('fielddate', 'app_platform', 'device_id')
        .sort('event_timestamp')
        .reduce(reducer)
        .join(get_previous_orders(job), by='puid', type='left', assume_unique_right=True)
        .join(
            job.table('home/passport/production/cards/uids_has_cards/' + end_date)
                .project('has_cards', puid=ne.custom(lambda uid: str(uid))),
            by='puid',
            type='left',
            assume_unique_right=True)
        .project(
            ne.all(),
            app_platform=ne.custom(lambda app_platform: ['_total_', app_platform]),
            has_cards=ne.custom(lambda has_cards: ['_total_'] + [False if has_cards is None else has_cards]),
            has_orders=ne.custom(
                lambda min_date, fielddate: ['_total_', str(min_date is not None and min_date < fielddate)]))
        .project(
            'device_id', 'puid', 'fielddate',
            'show', 'click', 'open', 'card',
            'hold_open', 'column', 'button', 'order',
            has_orders=se.unfold('has_orders_unfolded', 'has_orders'),
            app_platform=se.unfold('app_platform_unfolded', 'app_platform'),
            authorized=se.unfold('authorized_unfolded', 'authorized'),
            auto_open=se.unfold('auto_open_unfolded', 'auto_open'),
            has_cards=se.unfold('has_cards_unfolded', 'has_cards'),
            source=se.unfold('source_unfolded', 'source'))
        .groupby('fielddate', 'app_platform', 'authorized', 'auto_open', 'source', 'has_orders', 'has_cards') \
        .aggregate(
            show=na.sum('show'),
            click=na.sum('click'),
            open=na.sum('open'),
            card=na.sum('card'),
            hold_open=na.sum('hold_open'),
            column=na.sum('column'),
            button=na.sum('button'),
            order=na.sum('order'))
        .project(
            ne.all(),
            conversion_click=ne.custom(
                lambda show, click, source:
                get_conv(click, show) if source != 'Navi' and show > 0 else None),
            conversion_open=ne.custom(
                lambda show, open, source:
                get_conv(open, show) if source != 'Navi' and show > 0 else None),
            conversion_card=ne.custom(
                lambda show, card, source:
                get_conv(card, show) if source == 'Taximeter' and show > 0 else 1),
            conversion_hold_open=ne.custom(
                lambda show, hold_open, card, source:
                get_conv(hold_open, show) if source == 'Taximeter' else get_conv(hold_open, card)),
            conversion_column=ne.custom(
                lambda show, card, column, source:
                get_conv(column, show) if source == 'Taximeter' else get_conv(column, card)),
            conversion_button=ne.custom(
                lambda show, card, button, source:
                get_conv(button, show) if source == 'Taximeter' else get_conv(button, card)),
            conversion_order=ne.custom(
                lambda show, card, order, source:
                get_conv(order, show) if source == 'Taximeter' else get_conv(order, card)),

            conversion_prev_click=ne.custom(
                lambda show, click, source:
                get_conv(click, show) if source != 'Navi' else None),
            conversion_prev_open=ne.custom(
                lambda click, open, source:
                get_conv(open, click) if source != 'Navi' else None),
            conversion_prev_card=ne.custom(
                lambda open, card, source:
                get_conv(card, open) if source == 'Taximeter' else 1),
            conversion_prev_hold_open=ne.custom(
                lambda card, hold_open:
                get_conv(hold_open, card)),
            conversion_prev_column=ne.custom(
                lambda hold_open, column:
                get_conv(column, hold_open)),
            conversion_prev_button=ne.custom(
                lambda column, button:
                get_conv(button, column)),
            conversion_prev_order=ne.custom(
                lambda button, order:
                get_conv(order, button)))
        .publish(get_report(statface_client, options.scale))
    )
    return job


if __name__ == '__main__':
    cli.run()
