# -*- coding: utf-8 -*-
# pylint: disable=too-many-arguments
# pylint: disable=invalid-name
# pylint: disable=too-many-instance-attributes
# pylint: disable=too-many-locals

from __future__ import print_function, unicode_literals

from types import GeneratorType
from collections import OrderedDict as oDict, namedtuple
import re
import sys
import logging
import six
import requests
import itertools

from chapi.tools import (dump_request_error, _defaults, take,
                         get_iter_length, integers, safefetch, plain as _plain,
                         ProgIter, Delayer, TmpTable)
from chapi.errors import InternalError, ProgrammingError, OperationalError
from chapi.converter import FORMAT

logger = logging.getLogger('clickhouse_api')

_tee = itertools.tee(integers())[1]
_teetype = type(_tee)
del _tee


def _isgen(obj):
    return isinstance(obj, (GeneratorType, _teetype))


class Cursor(object):
    """Cursor class for accessing ClickHouse
    :param connection: Connection - parent connection instance
    :param jupyter: bool - use jupyter progressbar?
    :param arraysize: int - default chunk size in fetchmany
    :param save_backup: bool - save backup for query?
    """
    Ignored = namedtuple('Ignored', 'exc,query')
    def __init__(self, connection, jupyter=True,
                 arraysize=1, save_backup=True):
        self._connection = connection
        self.jupyter = jupyter
        self.save_backup = save_backup
        self._response_content = []
        self._size = 0
        self.arraysize = arraysize
        self._backup = []
        self._errors = []

    @property
    def connection(self):
        """Connection, used by cursor"""
        return self._connection

    @property
    def rowcount(self):
        """Count total number of rows if possible"""
        return self._size

    @property
    def backup(self):
        """Last result"""
        if not isinstance(self._backup, list):
            self._backup = list(self._backup)
        return self._backup

    def _split_and_save(self, content, gen_):
        if self.save_backup:
            backup, resulting = itertools.tee(content, 2)
            self._backup = backup
        else:
            resulting = content
        if not gen_:
            resulting = list(resulting)
        self._response_content = resulting

    def _execute(self, query, oldstyle_=False, httpargs=None, **kwargs):
        """Inner function for making queries, used by execute and executeiter

        :param query: str - your clickhouse query with formatting fields
        :param oldstyle_: bool, default False - use `%` formatting
            or `.format`?
        :param httpargs: dict - other connection params
        :param kwargs: key word args to format a query
        :return: generator over records
        """
        if oldstyle_:
            query = query % kwargs
        else:
            query = query.format(**kwargs)
        try:
            data = self._connection.request(
                    query=query,
                    httpargs=httpargs,
                    _output=True)
            self._size += len(data)
            self._errors.append(None)
        except OperationalError as e:
            self._errors.append(self.Ignored(e, query))
            data = []
        return (row for row in data)

    def execute(self, query, gen_=True, oldstyle_=False,
                httpargs=None, **kwargs):
        """Format and send request to ClickHouse

        :param query: str - your clickhouse query with formatting fields
        :param gen_: bool, default True - return generator over rows or a list?
        :param oldstyle_: bool, default False - use `%` formatting
            or `.format`?
        :param httpargs: dict - other connection params
        :param kwargs: key word args to format a query
        :return: generator | list
        """
        self._prepare()
        content = self._execute(query, oldstyle_=oldstyle_,
                                httpargs=httpargs, **kwargs)
        self._split_and_save(content, gen_)
        self.raise_for_errors()

    def executeiter(self, query, iterable, workers=1, gen_=True,
                    oldstyle_=False, trace=True, httpargs=None,
                    ordered=True, onerrors='raise'):
        """Iterate over iterable mappings, format a query with them end
        execute one by one

        :param query: str - your clickhouse query with formatting fields
        :param iterable: list(dict) - list of mappings for query
        :param workers: int, default 1 - desired num threads
        :param gen_: bool, default True - return generator over rows or a list?
            if gen_==True: then computes chunks on demand
        :param oldstyle_: bool, default False - use % formatting or .format?
        :param trace: - bool, default True - show progressbar?
        :param httpargs: dict - other connection params
        :param onerrors: str in {'raise', 'ignore'} - what to do on errors(after retries)?
        :return: generator | list
        """
        assert onerrors in {'raise', 'ignore'}, \
            "%s not in {'raise', 'ignore'}" % onerrors
        self._prepare()
        total = get_iter_length(iterable)
        from multiprocessing.dummy import Pool
        if oldstyle_:
            tasks = (query % it for it in iterable)
        else:
            tasks = (query.format(**it)
                     for it in iterable)
        pool = Pool(workers)
        imap_ = pool.imap_unordered
        if ordered:
            imap_ = pool.imap
        def execute(query):
            return self._execute(query, httpargs=httpargs)
        chunks = ProgIter(imap_(execute, tasks), total=total,
                          trace=trace, jupyter=self.jupyter)
        content = (row for chunk in chunks for row in chunk)
        self._split_and_save(content, gen_)
        if onerrors == 'raise':
            self.raise_for_errors()
        else:
            pass

    def execute_silent(self, query, httpargs=None):
        """Use this type of query when you don't expect output,
        e.g. drop table"""
        self._connection.request(query=query, _output=False,
                                 httpargs=httpargs)

    def fetchall(self, plain=False):
        """Returns all records

        :param plain: bool default False - return list or OrderedDict?
        :return: generator | list
        """
        res = self._response_content
        self._response_content = []
        self._size = 0
        if plain:
            if _isgen(res):
                res = (_plain(row) for row in res)
            else:
                res = [_plain(row) for row in res]
        return res

    @safefetch
    def fetchone(self, plain=False):
        """Returns one record

        :param plain: bool default False - return list or OrderedDict?
        :return: OrderedDict|list
        """
        if _isgen(self._response_content):
            item = next(self._response_content)
        else:
            item = self._response_content.pop(0)
        self._size -= 1
        if self._size < 0:
            self._size = 0
        if plain:
            item = _plain(item)
        return item

    def fetchmany(self, size=None, plain=False):
        """Returnes list of `size` records

        :param plain: bool default False - return list or OrderedDict?
        :return: list[OrderedDict|list]
        """
        size = size or self.arraysize
        if _isgen(self._response_content):
            result = take(size, self._response_content)
        else:
            result = self._response_content[:size]
            self._response_content = self._response_content[size:]
        if plain:
            result = list(map(_plain, result))
        self._size -= len(result)
        if self._size < 0:
            self._size = 0
        return result

    @property
    def errors(self):
        """Access to errors for user"""
        return self._errors[:]

    def has_errors(self):
        """Indicates if last execution had errors"""
        return any(self._errors)

    def raise_for_errors(self):
        """Raises the first met Exception"""
        errs = list(filter(bool, self._errors))
        if errs:
            raise errs[0].exc

    def _prepare(self):
        """Preparations for execution"""
        self._size = 0
        del self._errors[:]


# noinspection SqlNoDataSourceInspection
class Connection(object):
    """Factory for configured cursors

    :param host: str - ClickHouse host
    :param port: str - ClickHouse port
    :param username: str - ClickHouse Login
    :param password: str - ClickHouse Password
    :param connect_timeout: int
    :param read_timeout: int
    :param json: str - path to configuration json file
    :param jupyter: bool - if you want to see jupyter style progress bars
    :param errors: list[int] - memorize error codes
    :param raise_on_known: bool - raise on memorized error?
    :param retries: int - desired number of retries
    :param strategy: str in {'const', 'lin', 'poly', 'exp'} - delay strategy
    :param power: float|int - delay power
    """
    _httpargs = None
    def __init__(self, host=None, port=None, username=None, password=None,
                 connect_timeout=None, read_timeout=None,
                 json=None, jupyter=True,
                 retries=None,
                 errors=None, raise_on_known=None, strategy=None, power=None):
        defaults = _defaults(json)
        # must have params
        self.host = host if host is not None else defaults.get('host', None)
        self.port = port if port is not None else defaults.get('port', None)
        self.username = username if username is not None else defaults.get('username', None)
        self.password = password if password is not None else defaults.get('password', None)
        if (self.host is None or self.port is None or
                self.username is None or self.password is None):
            message = """
        Not all required arguments are passed:
        {%s}
        """ % " ,".join(arg for arg in ['host', 'port', 'username', 'password']
                        if getattr(self, arg) is None)
            raise ValueError(message)

        # non required params
        self.connect_timeout = (connect_timeout or
                                defaults.get('connect_timeout', 60))

        self.read_timeout = (read_timeout or
                             defaults.get('read_timeout', 60))

        self.retries = retries or defaults.get('retries', 0)

        self.errors = errors or defaults.get('errors', ())

        self.raise_on_known = (raise_on_known or
                               defaults.get('raise_on_known', False))

        strategy = strategy or defaults.get('strategy', 'lin')

        power = power or defaults.get('power', 2)

        self.delayer = Delayer(n=self.retries, c=power, strategy=strategy)

        self.jupyter = jupyter
        self._closed = False

    def set_httpargs(self, args):
        self._httpargs = args

    def request(self, query, _output=True, httpargs=None):
        """Performs request to clickhouse and returns list of dicts if output is needed

        :param query: str - query to execute
        :param _output: bool - do you expect any result?
        :param httpargs: dict - other connection params
        :return: dict | None
        """
        httpargs = httpargs or self._httpargs
        if self._closed:
            raise InternalError('Connection is closed')
        if re.search(r'[)\s]FORMAT\s', query, re.IGNORECASE):
            raise ProgrammingError('Formatting is not available')
        logger.info('%s@%s: %s', self.username, self.host, query)
        if _output:
            if isinstance(query, six.binary_type):
                query = '%s FORMAT JSONCompact' % query.decode()
            else:
                query = '%s FORMAT JSONCompact' % query
        with dump_request_error():
            left_retries = iter(self.delayer)
            response = self._attempt(query, left_retries, 1, httpargs)
            logger.info(
                'status code: %s \nheaders: %s \ncookies: %s \nsize: %d',
                response.status_code,
                response.headers,
                response.cookies.get_dict(),
                len(response.content)
            )
        if _output:
            try:
                json_result = response.json()
            except ValueError:
                raise OperationalError('Response is bad formatted, '
                                       'it can be if is too big')
            return self._prepare_output(json_result)

    @staticmethod
    def _prepare_output(content):
        addinfo = {k: v for k, v in content.items() if k != 'data'}
        try:
            names_types = [(d['name'], d['type']) for d in addinfo['meta']]
            def format_row(row):
                return [(nt[0], FORMAT[nt[1]](item))
                        for nt, item in zip(names_types, row)]
        except KeyError:
            def format_row(row):
                return [(i, item)
                        for i, item in zip(integers(), row)]
        return [oDict(format_row(row)) for row in content['data']]

    @property
    def server_uri(self):
        """http://{host}:{port}"""
        return 'http://{host}:{port}'.format(host=self.host, port=self.port)

    def cursor(self, **kwargs):
        """Get configured cursor

        Usage:
        ------
        >> conn = Connection()
        >> cursor = conn.cursor()

        :return: Cursor instance
        """
        if self._closed:
            raise InternalError('Connection is closed')
        defaults = dict(jupyter=self.jupyter)
        defaults.update(**kwargs)
        return Cursor(self, **defaults)

    def close(self):
        """Close the connection"""
        self._closed = True

    def _request(self, query, httpargs=None):
        """Useful for debugging if something really strange happens
        and you want to analyse response directly"""
        params = {
            'user': self.username,
            'password': self.password
        }
        params.update(httpargs or dict())
        return requests.post(
                self.server_uri,
                data=query.encode('utf-8'),
                params=params,
                timeout=(self.connect_timeout, self.read_timeout)
        )

    def _attempt(self, query, left_retries, n, httpargs=None):
        """Recursive method for handling ClickHouse exceptions

        :param query: str
        :param left_retries: iterable of ints
        :param n: attempt number
        :param httpargs: dict - other connection params
        :return: response from ClickHouse
        """
        import time
        response = self._request(query, httpargs)
        try:
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            if e.response is not None:
                e = OperationalError.from_response(e.response)
            else:
                e = OperationalError(str(e) + str('\n'))
            known = e.code in self.errors
            raise_on_known = self.raise_on_known
            ok = known ^ raise_on_known
            if ok:
                try:
                    time_to_wait = next(left_retries)
                    sys.stderr.write("failed %d times, wait %d seconds\r" %
                                     (n, time_to_wait))
                    time.sleep(time_to_wait)
                    return self._attempt(query, left_retries, n + 1, httpargs)
                except StopIteration:
                    exc = OperationalError(str('Retries: %d; %s') %
                                           (self.retries, str(e)))
                    raise exc
            else:
                raise e
        return response

    def gcollect(self, database, prefix='tmp_'):
        """Drops tables that start with `prefix` from database `db`

        :param database: str - database where tmp tables are located
        :param prefix: str - prefix for tmp tables

        .. Note
            It is supposed to use a database for safety,
            you should create it by yourself
        """
        cursor = self.cursor()
        cursor.execute("show tables from %s" % database)

        def choice(r):
            return r['name'].startswith(prefix)
        for row in filter(choice, cursor.fetchall()):   # type: dict
            self.request('DROP TABLE IF EXISTS {table}'.format(
                    table='%s.%s' % (database, row['name'])), _output=False)

    def tmp(self, select, db=None):
        """If you want to use tmp tables, use this thing
            Usage:
            ------
            >>  with conn.tmp(HEAVY_QUERY, 'mydb') as tablename:
            ...     # do your staff
            """
        return TmpTable(self, select, db)


def connect(*args, **kwargs):
    """shortcut for cursor"""
    return Connection(*args, **kwargs).cursor()
