# -*- coding: utf-8 -*-

import requests
import json
import collections
import urllib2
import io
import tarfile
import os
import commands
import logging
import pandas as pd
import re
import datetime

PATH = "//home/zaringleb/panels/"
CLIENTS_FROM_CONF = {}
logging.basicConfig(format='%(asctime)s %(levelname)s:%(message)s', filename=PATH + 'mylog.log',level=logging.INFO)


class Testid():
    '''Download, parse and store information about testid'''
    def __init__(self, testid):
        self.testid = testid
        self.get_testid_info()

    def get_testid_info(self):
        url = 'http://ab.yandex-team.ru/api/testid?type=ABT&form=full&id='
        req = requests.get(url + self.testid)
        self.config = json.loads(req.json()[0]['params'])[0] #json.loads(req.json()[0]['params'])
        self.parse_testid()

    def parse_testid(self):
        if self.config['HANDLER'] == 'REPORT':
            self.client = 'distr_serp'
            self.parse_report()
        elif self.config['HANDLER'] in ['GATEWAY', 'ATOM']:
            self.client = self.config['CONDITION'][len("SESSION_atom_client == '"): -1]
            self.parse_context()

    def parse_context(self):
        if 'atom.params.relev' not in self.config['CONTEXT'][self.config['HANDLER']] and self.client == 'promolib':
            self.sub_client = ""
        relev = self.config['CONTEXT'][self.config['HANDLER']]['atom.params.relev'][0]
        self.parse_relev()

    def parse_report(self):
        self.sub_client = None
        for sub_client in CLIENTS_FROM_CONF[self.client]:
            if ("atom_" + sub_client) in str(self.config):
                self.sub_client = sub_client

    def parse_relev(self):
        self.sub_client = None
        for sub_client in CLIENTS_FROM_CONF[self.client]:
            if ("atom_" + sub_client) in self.config['CONTEXT'][self.config['HANDLER']]['atom.params.relev'][0]:
                self.sub_client = sub_client
        if self.testid in ['48407', '48408', '48409', '48410', '48411', '48412', '48413', '48414', '48415', '48416', '48417', '48418']:
            self.sub_client = 'popup'

class Experiment():
    '''Contain information about experiment'''
    def __init__(self, task_info):
        self.ticket = task_info['ticket']
        self.authorities = task_info['authorities']
        self.author = task_info['author']
        self.str_testids = task_info['testids']

    def process_exp(self):
        self.testids = [Testid(str(testid)) for testid in self.str_testids]
        self.client = max([one.client for one in self.testids])
        self.sub_client = max([one.sub_client for one in self.testids])
        if not self.sub_client and self.client != 'promolib':
            raise Exception('empty subclient', self.ticket)
        self.host = self.get_host_by_client()
        self.get_ticket_info()

    def get_host_by_client(self):
        try:
            df_clients_hosts = pd.read_csv(PATH + 'clients_hosts.csv', sep=',')
            hosts = df_clients_hosts.loc[(df_clients_hosts['client'] == self.client)&(df_clients_hosts['sub_client'] == self.sub_client), 'distr_obj']
            return hosts.values[0]
        except Exception:
            print('Unable get host by client')
            return "_total_"

    def get_ticket_info(self):
        oauth = get_oauth()
        headers = {"Authorization": "OAuth {}".format(oauth)}
        r = requests.get("https://st-api.yandex-team.ru/v2/issues/{ticket}".format(ticket=self.ticket), headers=headers)
        q = json.loads(r.content)
        description = q['description'].encode('UTF-8')
        self.dates = self.get_dates(description)

    def get_dates(self, description):
        try:
            s = r'Даты проведения: \d\d\.\d\d\.\d\d\d\d-\d\d\.\d\d\.\d\d\d\d'
            w = re.search(s, description)
            begin_date, end_date = w.group()[len('Даты проведения: '):].split("-")
            begin_date = datetime.date(*[int(one) for one in begin_date.split('.')][::-1])
            end_date = datetime.date(*[int(one) for one in end_date.split('.')][::-1])
            return begin_date, end_date
        except Exception:
            print('Unable parse dates')

    def ticket_in_past_tickets(self):
        with open(PATH + 'tickets.txt') as f:
            tickets = f.readlines()
            tickets = [ticket.strip() for ticket in tickets]
        return self.ticket in tickets

    def put_in_past_tickets(self):
        with open(PATH + 'tickets.txt', 'a') as f:
            f.write(self.ticket + "\n")

def get_clients_from_conf():
    url = 'http://pecheny.haze.yandex.net:8049/viewconfig'
    req = requests.get(url)
    conf = req.json()
    clients_sub_clients = collections.defaultdict(list)
    for client in conf['clients']:
        if 'sub' in conf['clients'][client]:
            for sub_client in conf['clients'][client]['sub']:
                clients_sub_clients[client.lower()].append(sub_client.lower())
        else:
            clients_sub_clients[client.lower()].append("")
    return clients_sub_clients

def get_tasks(aspects=['atom', 'promolib', 'atom_int']):
    url = 'http://ab.yandex-team.ru/api/task?form=full&aspect='
    tasks = []
    for aspect in aspects:
        req = requests.get(url + aspect)
        #print(aspect)
        tasks.extend([one for one in req.json() if one['state'] in ['RUNNING', 'IN_CONFIG']])
    q = [{prop : one[prop] for prop in ['ticket', 'testids', 'authorities', 'author']} for one in tasks]
    return q

def plot_golovan(exp, only_str=False):
    os.chdir("/home/zaringleb/rerank_service/tools/scripts/yasm")

    command = 'python //home/zaringleb/rerank_service/tools/scripts/yasm/gen_atom_panels.py' \
                ' -u zaringleb --cc {editors} -c {client}{sub_client} -e "{ticket} {testids}"'.format(
                editors = ",".join(set(exp.authorities.split() + [exp.author] + ['zaringleb'])),
                client = exp.client,
                sub_client = "-" + exp.sub_client if exp.sub_client else "",
                ticket = exp.ticket,
                testids = ",".join([testid.testid for testid in exp.testids]))
    #logging.debug("command:{}".format(command))
    #print("command:{}".format(command))
    if only_str:
        print(command)
    else:
        status, output = commands.getstatusoutput(command)
        plot_id = json.loads(output.split("\n", 1)[-1][len("INFO:root:"):])['result']['_id']
        golovan_url = "https://yasm.yandex-team.ru/panel/" + plot_id
        return golovan_url

def get_oauth():
    with open('/home/zaringleb/.oauth') as f:
        return f.read().rstrip()

def get_atomdistribution_url(exp):
    init_url = 'https://stat.yandex-team.ru/Distribution/Others/AtomBanners/AtomDistribution'
    params = ['_incl_fields=shows',
              '_incl_fields=trueinstalls',
              '_incl_fields=trueinstallsrate',
              'scale=d',
              'product=_total_',
              'referer=_total_',
              'region=_total_',
              'browser=_total_',
              'os=_total_']

    params.append('distr_obj={}'.format(exp.host))
    params.extend(["testid={}".format(testid) for testid in exp.str_testids])
    if exp.dates:
        params.append("date_min={}+00%3A00%3A00".format(exp.dates[0]))
        params.append("date_max={}+23%3A59%3A59".format(exp.dates[1]))
    else:
        params.append('_period_distance=7')
    atomdistribution_url = "{}?{}".format(init_url, "&".join(params))
    return atomdistribution_url

def post_in_ticket(messages, ticket):
    oauth = get_oauth()
    headers = {"Authorization": "OAuth {}".format(oauth), "Content-Type": "application/json; charset: UTF-8"}
    text = "\n".join(['(({} {}))'.format(messages[key], key) for key in messages])
    ticket_data = { "text": text, "summonees" : ["zaringleb"]}
    r = requests.post("https://st-api.yandex-team.ru/v2/issues/{ticket}/comments".format(ticket=ticket), headers=headers, data=json.dumps(ticket_data))

def main():
    global CLIENTS_FROM_CONF
    logging.info('Start')
    CLIENTS_FROM_CONF = get_clients_from_conf()
    logging.debug('len(atom_conf):{}'.format(len(CLIENTS_FROM_CONF)))
    tasks = get_tasks()
    for task in tasks:
        try:
            exp = Experiment(task)
            if not exp.ticket_in_past_tickets():
                exp.process_exp()
                plot_id = plot_golovan(exp)
                v4_url =  get_atomdistribution_url(exp)
                post_in_ticket({'AtomDistribution' : v4_url, 'golovan' : plot_id}, exp.ticket)
                exp.put_in_past_tickets()
        except Exception as e:
            print(e, task['ticket'])
if __name__ == "__main__":
    main()
