#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from __future__ import division
import sys
import os
import codecs
import argparse
import openpyxl
from collections import defaultdict
import itertools
import datetime
import random
from copy import deepcopy
import json
import requests
from prepare_atom_experiment import get_creatives, post_creatives
from pytils import make_logger


rnd = random.SystemRandom()


def make_landing(dct, type_='wizard'):
    link = ('extension_link' if type_ == 'wizard' else 'link')
    dct['aux-data']['_link_extensions'][link].update(
        {
            "banner_id": "$banner_id",
            "host": "$host",
            "reqid": "reqid"
        }
    )
    dct['aux-data'].update(
        {
            "type": "landing"
        }
    )
    dct.update(
        {
            "type": "landing"
        }
    )


def make_download(dct, type_='wizard'):
    if type_ == 'wizard':
        dct['aux-data'].update(
            {
                "install_description":
                    "После этого начнётся установка Яндекс.Браузера",
                "install_text": "Нажмите на стрелку и выберите верхний файл",
                "legal": {
                    "link": "https://yandex.ru/legal/browser_agreement",
                    "text": "Устанавливая Яндекс.Браузер, вы принимаете условия лицензионного соглашения"
                },
            }
        )
    elif type_ == 'popup':
        dct['aux-data'].update(
            {
                "legal": {
                    "link": "https://yandex.ru/legal/browser_agreement",
                    "linkText": "условия лицензионного соглашения",
                    "text": "Устанавливая Яндекс Браузер, вы принимаете "
                },
                "install_text": "Нажмите${link_text}на кнопку, на которую указывает стрелка",
            }
        )
    dct['aux-data'].update(
        {
            "type": "download",
        }
    )
    dct.update(
        {
            "type": "download",
        }
    )


def make_landing_or_download(param, dct, type_='wizard'):
    link = ('extension_link' if type_ == 'wizard' else 'link')
    if param[0] == 'landing':
        make_landing(dct, type_=type_)
    elif param[0] == 'download':
        make_download(dct, type_=type_)
    dct['link'] = param[1]
    dct['aux-data'][link] = param[1]


def to_values(x):
    return tuple(y.value for y in x)


def read_worksheet(worksheet):
    result = []
    header = to_values(worksheet.rows[0])
    for row in worksheet.rows[1:]:
        result.append(dict(zip(header, to_values(row))))
    return result


def apply_browser(dct, browser_dct):
    dct['filter'] += '&{}'.format(browser_dct['uat'])
    dct['aux-data']['guide_image'] = browser_dct['guide_image']
    dct['aux-data']['position'] = browser_dct['position']


class InternalUrlGenerator(object):

    def __init__(self):
        self.id_ = 0

    def __call__(self):
        ts = datetime.datetime.now().strftime('%s')
        result = ts + str(self.id_)
        self.id_ += 1
        return result


internal_url_generator = InternalUrlGenerator()


def generate_wizard(config, template, browser_config):
    result = defaultdict(lambda: [])
    for product in config:
        for feature in config[product]:
            dct = config[product][feature]
            links = []
            for param in ('landing', 'download'):
                links.extend(
                    [(param, x) for x in dct['link_{}'.format(param)]]
                )
            for comb in itertools.product(
                [x for x in dct['text'] if len(x) <= 62],
                links,
                dct['image'],
                dct['yes'],
                dct['no'],
                # browser_config
            ):
                new = deepcopy(template)
                new['title'] = comb[0]
                new['url'] = 'default_search/{}'.format(product)
                new['__product'] = product
                make_landing_or_download(comb[1], new)
                img_link = '//yabs.yandex.ru/resource/{}'.format(comb[2])
                new['image'] = img_link
                new['aux-data']['image'] = img_link
                new['aux-data']['image_retina'] = img_link
                new['internal-url'] = 'default_search/{}'.format(
                    internal_url_generator()
                )
                # apply_browser(new, comb[5])
                new['yes'] = comb[3]
                new['aux-data']['button_yes'] = comb[3]
                new['no'] = comb[4]
                new['aux-data']['button_no'] = comb[4]
                result[feature].append(new)
    return result


def generate_popup(config, template, browser_config):
    result = defaultdict(lambda: [])
    for product in config:
        for feature in config[product]:
            dct = config[product][feature]
            links = []
            for param in ('landing', 'download'):
                links.extend(
                    [(param, x) for x in dct['link_{}'.format(param)]]
                )
            for comb in itertools.product(
                [x for x in dct['text'] if len(x) <= 62],
                links,
                dct['image'],
                dct['yes'],
                dct['no'],
                # browser_config
            ):
                new = deepcopy(template)
                new['title'] = comb[0]
                new['aux-data']['text'] = comb[0]
                new['url'] = 'default_search/{}'.format(product)
                new['__product'] = product
                make_landing_or_download(comb[1], new, type_='popup')
                img_link = ['//yabs.yandex.ru/resource/{}'.format(x)
                            for x in comb[2]]
                new['image'] = img_link
                for x in ['image', 'image_png', 'icon', 'icon_png']:
                    new['aux-data'][x] = img_link[0]
                for x in ['image_svg', 'icon_svg']:
                    new['aux-data'][x] = img_link[1]
                new['internal-url'] = 'portal_popup/{}'.format(
                    internal_url_generator()
                )
                # apply_browser(new, comb[5])
                new['yes'] = comb[3]
                new['aux-data']['button_yes'] = comb[3]
                new['no'] = comb[4]
                new['aux-data']['button_no'] = comb[4]
                result[feature].append(new)
    return result


def generate_popup_wrapper(filename):
    wb = openpyxl.load_workbook(filename)

    config = defaultdict(lambda: defaultdict(lambda: defaultdict(lambda: [])))

    texts = read_worksheet(wb.get_sheet_by_name('Тексты'))

    for rec in texts:
        if rec['Product'] and rec['Feature'] and rec['text']:
            config[rec['Product']][rec['Feature']]['text'].append(
                rec['text']
            )

    links = read_worksheet(wb.get_sheet_by_name('Ссылки'))

    for rec in links:
        if (rec['Product'] and
            rec['Feature'] and
            rec['Link'] and
                rec['Type']):
            config[rec['Product']][rec['Feature']][
                'link_{}'.format(rec['Type'])
            ].append(
                rec['Link']
            )

    images = read_worksheet(wb.get_sheet_by_name('Картинки попапы'))

    for rec in images:
        if (
            rec['Product'] and
            rec['Feature'] and
            rec['image png'] and
            rec['image svg']
        ):
            if rec['Feature'] != 'ALL':
                config[rec['Product']][rec['Feature']]['image'].append(
                    (rec['image png'], rec['image svg'])
                )
            else:
                for feature in config[rec['Product']]:
                    config[rec['Product']][feature]['image'].append(
                        (rec['image png'], rec['image svg'])
                    )

    buttons = read_worksheet(wb.get_sheet_by_name('Кнопки'))

    for rec in buttons:
        if rec['Product'] and rec['Type'] and rec['Text']:
            for feature in config[rec['Product']]:
                config[rec['Product']][feature][rec['Type']].append(
                    rec['Text']
                )

    result = generate_popup(
        config,
        json.load(open('popup.json')),
        json.load(open('browsers.json')),
    )
    print('number of generated popup creatives: {}'.format(count_len(result)))
    json.dump(
        dict(result),
        codecs.open('portal_popup_autogen.json', 'w', 'utf8'),
        ensure_ascii=False,
        indent=4,
        sort_keys=True
    )


def count_len(dct):
    result = 0
    for x in dct:
        result += len(dct[x])
    return result


def generate_wizard_wrapper(filename):
    wb = openpyxl.load_workbook(filename)

    config = defaultdict(lambda: defaultdict(lambda: defaultdict(lambda: [])))

    texts = read_worksheet(wb.get_sheet_by_name('Тексты'))

    for rec in texts:
        if rec['Product'] and rec['Feature'] and rec['text']:
            config[rec['Product']][rec['Feature']]['text'].append(
                rec['text']
            )

    links = read_worksheet(wb.get_sheet_by_name('Ссылки'))

    for rec in links:
        if (rec['Product'] and
            rec['Feature'] and
            rec['Link'] and
                rec['Type']):
            config[rec['Product']][rec['Feature']][
                'link_{}'.format(rec['Type'])
            ].append(
                rec['Link']
            )

    images = read_worksheet(wb.get_sheet_by_name('Картинки колдунщики'))

    for rec in images:
        if rec['Product'] and rec['Feature'] and rec['image png']:
            if rec['Feature'] != 'ALL':
                config[rec['Product']][rec['Feature']]['image'].append(
                    rec['image png']
                )
            else:
                for feature in config[rec['Product']]:
                    config[rec['Product']][feature]['image'].append(
                        rec['image png']
                    )

    buttons = read_worksheet(wb.get_sheet_by_name('Кнопки'))

    for rec in buttons:
        if rec['Product'] and rec['Type'] and rec['Text']:
            for feature in config[rec['Product']]:
                config[rec['Product']][feature][rec['Type']].append(
                    rec['Text']
                )

    result = generate_wizard(
        config,
        json.load(open('wizard.json')),
        json.load(open('browsers.json')),
    )
    print('number of generated wizard creatives: {}'.format(count_len(result)))
    json.dump(
        dict(result),
        codecs.open('distr_wizard_autogen.json', 'w', 'utf8'),
        ensure_ascii=False,
        indent=4,
        sort_keys=True
    )


def parse_val(dct):
    result = {}
    result['ts'] = dct['ts']
    result.update(
        {x.split('=')[0]: int(x.split('=')[1])
         for x in dct['v'].split()}
    )
    return result


def get_data_from_rtmr(list_='distr_wizard'):
    req = requests.get(
        'http://rtmr-sas.search.yandex.net:8080/yandsearch?'
        'view=json&table=atom/candidate_scores'
        '&key={}&maxrecords=1&maxtimeback=9999999&mrs=999999999'
        .format(list_)
    )
    obj = json.loads(req.json()['Entries'][0]['Value'])
    obj.pop(sorted(obj)[0])
    good_dump(obj, '{}/logs/rtmr_stats_{}.json'.format(
        os.path.dirname(os.path.abspath(__file__)),
        datetime.datetime.now().strftime('%Y%m%d%H%M')
    ))
    return parse_rtmr_answer(obj)


def parse_rtmr_answer(obj):
    return {k: parse_val(obj[k]) for k in obj}


def diff_values(dict_a, dict_b):
    result = {'ts': dict_a['ts']}
    for key in dict_a:
        if key in dict_b:
            result[key] = dict_a[key] - dict_b[key]
    return result


def diff_dicts(current, previous):
    result = {}
    for key in current:
        if key in previous:
            result[key] = diff_values(current[key], previous[key])
    return result


def good_dump(obj, filename):
    json.dump(
        obj,
        codecs.open(filename, 'w', 'utf8'),
        sort_keys=True,
        ensure_ascii=False,
        indent=4
    )


def get_rtmr_diff(list_='distr_wizard'):
    previous = {}
    previous_filename = '{}_stats.json'.format(list_)
    if os.path.isfile(previous_filename):
        previous = json.load(open(previous_filename))
    current = get_data_from_rtmr(list_=list_)
    absolute_copy = deepcopy(previous)
    for key in current:
        absolute_copy[key] = current[key]
    json.dump(
        absolute_copy,
        codecs.open(previous_filename, 'w', 'utf8'),
        ensure_ascii=False,
        sort_keys=True,
        indent=4
    )
    return diff_dicts(current, previous)


def int_split(int_, n_parts):
    even = int_ // n_parts
    result = [even] * n_parts
    remainder = int_ % n_parts
    mover = 0
    while remainder > 0:
        if mover == len(result):
            mover = 0
        result[mover] += 1
        remainder -= 1
        mover += 1
    return result


# snippet adapted from http://stackoverflow.com/a/4463433/4328153
def make_unique(seq):  # Order preserving
    ''' Modified version of Dave Kirby solution '''
    seen = set()
    return [x for x in seq if x not in seen and not seen.add(x)]
# end of snippet


def remove_subst(s):
    if '_SUBST' in s:
        s = s.split('_SUBST')[0]
    return s


def sorted_good(dct):
    return make_unique([
        remove_subst(x)
        for x in sorted(
            dct,
            key=lambda y: (
                (
                    dct[y].get('SOFT_EXPORT_EVENT_install_d:3', 0) +
                    dct[y].get('SOFT_EXPORT_EVENT_dayuse_d:3', 0)
                ),
                dct[y].get('show_d:3', 0)
            ),
            reverse=True)
    ])


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('filename')
    parser.add_argument('--first', action='store_true')
    parser.add_argument('--debug', action='store_true')
    args = parser.parse_args()

    logger = make_logger(os.path.abspath(__file__), debug=args.debug)

    if os.path.isfile(args.filename):
        if not os.path.isfile('distr_wizard_autogen.json'):
            generate_wizard_wrapper(args.filename)
        if not os.path.isfile('portal_popup_autogen.json'):
            generate_popup_wrapper(args.filename)
    elif args.filename == 'regular':
        for place in [
            'distr_wizard_autogen',
            # 'portal_popup_autogen'
        ]:
            creatives = get_creatives(place)
            creatives_inturls = {x['internal-url'] for x in creatives}
            diff = get_data_from_rtmr(list_=place)
            if args.first:
                continue
            diff_sorted = sorted_good(diff)
            diff_top150 = []
            for x in diff_sorted:
                if x in creatives_inturls and x not in diff_top150:
                    diff_top150.append(x)
                if len(diff_top150) == 150:
                    break
            if len(diff_top150) < 150:
                logger.error(
                    'Less than 150 internal-urls in top, probably rtmr hiccups'
                )
                if args.debug:
                    import pdb; pdb.set_trace()
                sys.exit(1)
            top150 = [
                x for x in creatives
                if x['internal-url'] in diff_top150
            ]
            logger.info('top 150: {}'.format(', '.join(
                ['{}: {}'.format(e, x) for e, x in enumerate(diff_top150)]
            )))
            deleted = sorted([
                x['internal-url'] for x in creatives
                if x['internal-url'] not in diff_top150
            ])
            logger.info('deleted: {}'.format(', '.join(
                ['{}: {}'.format(e, x) for e, x in enumerate(deleted)]
            )))
            obj = json.load(open('{}.json'.format(place)))
            features = list(obj.keys())
            rnd.shuffle(features)
            feat_split = int_split(300 - len(top150), len(features))
            for feature in features:
                rnd.shuffle(obj[feature])
                top150.extend(obj[feature][:feat_split[0]])
                logger.info('adding feature {}: {}'.format(
                    feature, ', '.join([
                        x['internal-url'] for x in obj[feature][:feat_split[0]]
                    ])
                ))
                obj[feature] = obj[feature][feat_split[0]:]
                feat_split = feat_split[1:]
            if args.debug:
                import pdb
                pdb.set_trace()
            json.dump(
                obj,
                codecs.open('{}.json'.format(place), 'w', 'utf8'),
                ensure_ascii=False,
                sort_keys=True,
                indent=4
            )
            post_creatives(place, top150, reqtype='put')
    elif args.filename == 'populate':
        for place in [
            'distr_wizard_autogen', 'portal_popup_autogen'
        ]:
            creatives = []
            obj = json.load(open('{}.json'.format(place)))
            features = list(obj.keys())
            rnd.shuffle(features)
            feat_split = int_split(300, len(features))
            pass
            for feature in features:
                rnd.shuffle(obj[feature])
                creatives.extend(obj[feature][:feat_split[0]])
                obj[feature] = obj[feature][feat_split[0]:]
                feat_split = feat_split[1:]
            json.dump(
                obj,
                codecs.open('{}.json'.format(place), 'w', 'utf8'),
                ensure_ascii=False,
                sort_keys=True,
                indent=4
            )
            post_creatives(place, creatives)


if __name__ == "__main__":
    main()
