#!usr/bin/env python
#! -*- coding: utf-8 -*-
from __future__ import division
from __future__ import unicode_literals
import os
import sys
import re
import argparse
import codecs
import pdb
import traceback
import hashlib
from collections import Counter
import logging
import requests
import datetime
import time
import toml
from pecheny.moncommons import push_to_razladki

re_fr = re.compile(r'(?<=full-request=).*?(?=[\t$])')
re_login = re.compile(r'(?<=yandex_login=)[0-9a-zA-Z\-\.@]+')
re_query = re.compile(r'(?<=query=).*?(?=[\t$])')
re_msp = re.compile(r'(?<=msp=).*?(?=[\t$])')
isletter = re.compile(ur'[A-Za-zА-Яа-я]')


class Testcase(object):

    def __init__(self, id, query, debugtext):
        self.id = id
        self.query = query
        self.debugtext = debugtext


def yandex_login(s):
    if re_login.search(s):
        return re_login.search(s).group(0)
    return ''


def get_query(s):
    if re_msp.search(s):  # use corrected version of query, if exists
        return re_msp.search(s).group(0).split(':')[-1]
    elif re_query.search(s):
        return re_query.search(s).group(0)
    return ''


def full_request(s):
    if re_fr.search(s):
        return re_fr.search(s).group(0)
    return ''


def current_timestamp():
    return int((datetime.datetime.now()
                - datetime.datetime(1970, 1, 1)).total_seconds())


def fail(failures, reason, text):
    failures[reason] += 1
    return text

# def push_to_razladki(params, desc, value):
#     data = {desc: value}
#     req = None
#     while req is None or req.status_code != 200:
#         req = requests.post(params['razladki'], data=data)


def md5(string_):
    return hashlib.md5(string_).hexdigest()


def main():

    global __file__                         # to fix stupid
    __file__ = os.path.abspath(__file__)    # __file__ handling
    _file_ = os.path.basename(__file__)     # in python 2

    parser = argparse.ArgumentParser()
    parser.add_argument('--logins', '-l', type=int, default=100)
    parser.add_argument('--ystaff', '-y', action='store_true')
    parser.add_argument('--debug', '-d', action='store_true')
    parser.add_argument('--config', '-r', default=None)
    args = parser.parse_args()

    testcases = []
    badqueries = 0
    nonzeroes = 0
    nologins = 0

    now = datetime.datetime.now()
    start = current_timestamp()

    # set up logging
    logger = logging.getLogger(_file_[:-3])
    formatter = logging.Formatter('%(asctime)s | %(message)s')
    ch = logging.StreamHandler()
    logger.setLevel(logging.DEBUG)
    if args.debug:
        ch.setLevel(logging.DEBUG)
    else:
        ch.setLevel(logging.CRITICAL)
    ch.setFormatter(formatter)
    logger.addHandler(ch)
    fh = logging.FileHandler('{}/logs/{}-{}.log'.format(
        os.path.dirname(__file__), _file_[:-3], start),
        encoding='utf8')
    fh.setLevel(logging.DEBUG)
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    with open('basic.toml', 'r') as f:
        params = toml.loads(f.read())
    os.chdir(os.path.dirname(__file__))
    with open('geo.toml', 'r') as f:
        params.update(toml.loads(f.read()))
    if not args.config is None:
        with open(args.config, 'r') as f:
            params.update(toml.loads(f.read()))

    rtmr = params['rtmr']
    if args.ystaff:
        fetch_table = params['fetch_table_staff']
    else:
        fetch_table = params['fetch_table']
    check_table = params['check_table']
    rtmr_check = params['rtmr_check']

    failures = Counter()
    fetch_req = params['fetch_req'].format(rtmr, fetch_table)
    prevtext = ''

    while (len(testcases) < args.logins
           and sum(failures.values()) < 1000 * args.logins):
        try:
            if (current_timestamp() - start) > 3600:
                sys.exit(1)
            req = requests.get(fetch_req, timeout=1)
            text, exit_status = (req.content.decode('utf8', errors='replace'),
                                 req.status_code)
            if exit_status != 200:
                failures['request failed'] += 1
                continue
            if text == prevtext:
                failures['same line'] += 1
                time.sleep(1)
                continue

            if not 'service=maps.yandex' in text:
                prevtext = fail(failures, 'wrong service', text)
                continue
            if not 'type=REQUEST' in text:
                prevtext = fail(failures, 'not a request', text)
                continue
            if 'output=json' in text:
                prevtext = fail(failures, 'output=json', text)
                continue
            if not yandex_login(text):
                prevtext = fail(failures, 'non-login', text)
                continue
            if not (isletter.search(get_query(text))
                    and len(get_query(text)) > 2):
                prevtext = fail(failures, 'bad query', text)
                continue

            testcases.append(Testcase(id=yandex_login(text),
                                      query=get_query(text),
                                      debugtext=text))
            try:
                logger.info(u'pair {} [{}] added'.format(
                    testcases[-1].id, testcases[-1].query))
            except:
                logger.info(traceback.format_exc())
            prevtext = text
        except:
            prevtext = fail(failures, 'other error', '')
            logger.info(traceback.format_exc())
            continue

    mid = current_timestamp()
    total_seconds = mid - start
    logger.info('{} login-query pairs retrieved'
                ' from table {} in {:.2f} seconds ('
                '{:.2f} avg sec per try). Login rate is {:.2%}.'.format(
                    len(testcases), fetch_table, total_seconds,
                    total_seconds / len(testcases),
                    len(testcases) / (len(testcases) + failures['non-login'])))

    logger.info('Failure detalization: {}'.format(failures))

    found = 0
    notfound = 0
    nonzeroes = 0
    fails = 0

    while len(testcases) > 0 and (nonzeroes + fails) < 100 * args.logins:
        try:
            if (current_timestamp() - start) > 3600:
                sys.exit(1)
            check_req = params['check_req'].format(
                rtmr_check, check_table, md5(testcases[0].id))
            req = requests.get(check_req, timeout=1)
            text, exit_status = (req.content.decode('utf8', errors='replace'),
                                 req.status_code)
            if exit_status != 200:
                logger.info('request failed')
                nonzeroes += 1
                continue
        except:
            logger.info(traceback.format_exc())
            fails += 1
            continue
        try:
            if testcases[0].query in text:
                found += 1
            else:
                notfound += 1
                logger.info('not found: {}, [{}]. debug: {}.'
                            ' original logline: {}'.format(
                                testcases[0].id, testcases[0].query,
                                check_req, testcases[0].debugtext))
            testcases.pop(0)

        except:
            logger.info(traceback.format_exc())
            fails += 1
            testcases.pop(0)
            continue

    try:
        foundrate = found / (found + notfound)
    except:
        foundrate = 0

    logger.info('{} queries found in geo_requests ({:.1%}), '
                '{} queries not found, {} non-zero exit codes.'.format(
                    found, foundrate, notfound, nonzeroes))

    desc = 'geo_rtmr_foundrate'
    if 'suffix' in params:
        desc += params['suffix']
    if not args.debug:
        push_to_razladki(params, desc, foundrate)

if __name__ == '__main__':
    main()
