#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from __future__ import division
import sys
import os
import codecs
import datetime
import psutil
from mrdef import defaults
from putils import make_logger


def lt(x, y):
    try:
        return x < y
    except TypeError:
        return True


def inc_lt(x, y):
    try:
        return x <= y
    except TypeError:
        return True


class Monitoring(object):

    def __init__(self, prefix, lockpath, lastpath,
                 default_id, str_to_table_id, process_table,
                 mapreduce, get_srctables=None, filter_tables=None):
        self.prefix = prefix
        self.lockpath = lockpath
        self.lastpath = lastpath
        self.default_id = default_id
        self.str_to_table_id = str_to_table_id
        self.process_table = process_table
        self.MapReduce = mapreduce

        if get_srctables:
            self.get_srctables = get_srctables
        if filter_tables:
            self.filter_tables = filter_tables

    def lock(self):
        with codecs.open(self.lockpath, 'w', 'utf8') as f:
            f.write('Locked at {}'.format(datetime.datetime.now()))

    def unlock(self):
        with codecs.open(self.lockpath, 'w', 'utf8') as f:
            f.write('Free')

    def check_if_locked(self):
        try:
            with codecs.open(self.lockpath, 'r', 'utf8') as f:
                contents = f.read().rstrip()
        except IOError:
            contents = 'free'
            with codecs.open(self.lockpath, 'w', 'utf8') as f:
                f.write(contents)
        search_for_process = [p for p in psutil.process_iter()
                              if (os.path.basename(__file__)
                                  in ' '.join(p.cmdline())) and
                              ('nolock' not in ' '.join(p.cmdline())) and
                              ('mapreduce' not in ' '.join(p.cmdline()))]
        if 'locked' in contents and len(search_for_process) > 1:
            self.logger.info('Process is locked, exiting...')
            sys.exit(0)
        else:
            with codecs.open(self.lockpath, 'w', 'utf8') as f:
                f.write('locked at {}'.format(datetime.datetime.now()))

    def get_lasttable(self):
        try:
            with codecs.open(self.lastpath, 'r', 'utf8') as f:
                return self.str_to_table_id(f.read())
        except IOError:
            return self.str_to_table_id(self.default_id)

    def set_lasttable(self, table):
        with codecs.open(self.lastpath, 'w', 'utf8') as f:
            f.write(table)

    def get_srctables(self):
        ti = self.MapReduce.getTablesInfo('{}*'.format(self.prefix))
        return sorted([x.name for x in ti
                       if '/' not in x.name[len(self.prefix):]])

    def filter_tables(self, tables, lb=None, ub=None, inc=False):
        fn = lt
        if inc:
            fn = inc_lt
        return [x for x in tables if
                fn(lb, self.str_to_table_id(x)) and
                fn(self.str_to_table_id(x), ub)]

    def start(self, args, kwargs):
        _from = self.str_to_table_id(getattr(args, 'from'))
        _to = self.str_to_table_id(getattr(args, 'to'))

        self.logger = make_logger(__file__, debug=args.debug)
        kwargs['logger'] = self.logger

        if not args.nolock:
            self.check_if_locked()

        self.logger.info('Setting mapreduce defaults...')
        defaults()
        self.MapReduce.useDefaults(username='personalization')

        lasttable = self.get_lasttable()

        self.logger.info('Getting source tables...')
        if (not _from and not _to):
            srctables = self.filter_tables(self.get_srctables(),
                                           lb=lasttable)
            while srctables:
                self.process_table(srctables[0], **kwargs)
                self.set_lasttable(srctables[0])
                lasttable = self.get_lasttable()
                srctables = self.filter_tables(self.get_srctables(),
                                               lb=lasttable)
        else:
            srctables = self.filter_tables(self.get_srctables(),
                                           lb=_from, ub=_to, inc=True)
            for table in srctables:
                self.process_table(table, **kwargs)

        self.unlock()
