#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from __future__ import division
import sys
import os
import codecs
import datetime
import types
try:
    import psutil
except:
    pass
import yt.wrapper as yt
from pytils import (make_logger, yt_config_set_defaults, get_yt_exists,
                    yt_get_date_from_table, date_range)
yt_exists = get_yt_exists(yt)


def lt(x, y):
    if type(x) == type(y):
        return x < y
    else:
        return True


def inc_lt(x, y):
    if type(x) == type(y):
        return x <= y
    else:
        return True


def wrap_cmdline(s):
    return [x.decode('utf8', errors='replace') for x in s]


def nxt(d):
    return d + datetime.timedelta(days=1)


class Monitoring(object):

    def __init__(self, prefix, filepath,
                 default_id, process_table,
                 str_to_table_id=None, get_srctables=None, filter_tables=None,
                 logger=None, confirmation=False, today=True, mode='tables'):
        self.prefix = prefix
        self.filepath = filepath
        self.lockpath = filepath + '.lock'
        self.lastpath = filepath + '.last'
        self.filename = os.path.basename(filepath)
        self.default_id = default_id
        self.mode = mode
        self.today = today
        assert self.mode in {'tables', 'dates'}
        if self.mode == 'tables':
            assert str_to_table_id
            self.str_to_table_id = str_to_table_id
        elif self.mode == 'dates':
            self.str_to_table_id = lambda x: datetime.datetime.strptime(
                x, '%Y-%m-%d'
            ).date()
        self.process_table = process_table
        self.logger = logger
        self.confirmation = confirmation

        if get_srctables:
            self.get_srctables = types.MethodType(get_srctables, self)
        if filter_tables:
            self.filter_tables = types.MethodType(filter_tables, self)

    def lock(self):
        with codecs.open(self.lockpath, 'w', 'utf8') as f:
            f.write('Locked at {}'.format(datetime.datetime.now()))

    def unlock(self):
        with codecs.open(self.lockpath, 'w', 'utf8') as f:
            f.write('Free')

    def check_if_locked(self):
        try:
            with codecs.open(self.lockpath, 'r', 'utf8') as f:
                contents = f.read().rstrip()
        except IOError:
            contents = 'Free'
            with codecs.open(self.lockpath, 'w', 'utf8') as f:
                f.write(contents)
        search_for_process = [
            p for p in psutil.process_iter()
            if (self.filename
                in ' '.join(wrap_cmdline(p.cmdline()))) and
            ('nolock' not in ' '.join(wrap_cmdline(p.cmdline()))) and
            ('mapreduce' not in ' '.join(wrap_cmdline(p.cmdline())))
        ]
        if 'locked' in contents and len(search_for_process) > 1:
            self.logger.info('Process is locked, exiting...')
            sys.exit(0)
        else:
            with codecs.open(self.lockpath, 'w', 'utf8') as f:
                f.write('locked at {}'.format(datetime.datetime.now()))

    def get_lasttable(self):
        try:
            with codecs.open(self.lastpath, 'r', 'utf8') as f:
                return self.str_to_table_id(f.read().strip())
        except IOError:
            return self.str_to_table_id(self.default_id)

    def set_lasttable(self, table):
        with codecs.open(self.lastpath, 'w', 'utf8') as f:
            f.write(table)

    def get_srctables(self):
        ti = ['{}/{}'.format(self.prefix, x) for x in
              yt.list('{}'.format(self.prefix))]
        return sorted(ti)

    def filter_tables(self, tables, lb=None, ub=None, inc=False):
        fn = lt
        if inc:
            fn = inc_lt
        return [x for x in tables if
                self.str_to_table_id(x) and
                fn(lb, self.str_to_table_id(x)) and
                fn(self.str_to_table_id(x), ub)]

    def start(self, args, kwargs=None):
        if not kwargs:
            kwargs = {}

        try:
            _from = self.str_to_table_id(getattr(args, 'from'))
        except:
            _from = None
        try:
            _to = self.str_to_table_id(getattr(args, 'to'))
        except:
            _to = None

        if not self.logger:
            self.logger = make_logger(__file__, debug=args.debug)
        kwargs['logger'] = self.logger

        yt_config_set_defaults(yt, self.logger)

        if not args.nolock:
            self.check_if_locked()

        lasttable = self.get_lasttable()

        if self.today:
            thisdate = datetime.datetime.now().date()
        else:
            thisdate = (
                datetime.datetime.now().date() - datetime.timedelta(days=1)
            )

        if self.mode == 'tables':
            self.logger.info('Getting source tables...')
            if (not _from and not _to):
                srctables = self.filter_tables(self.get_srctables(),
                                               lb=lasttable)
                while srctables:
                    conf_marker = self.process_table(srctables[0], **kwargs)
                    if not self.confirmation or conf_marker:
                        self.set_lasttable(srctables[0])
                    lasttable = self.get_lasttable()
                    srctables = self.filter_tables(self.get_srctables(),
                                                   lb=lasttable)
            else:
                srctables = self.filter_tables(self.get_srctables(),
                                               lb=_from, ub=_to, inc=True)
                for table in srctables:
                    self.process_table(table, **kwargs)
        elif self.mode == 'dates':
            if (not _from and not _to):
                srctables = date_range(
                    nxt(lasttable), thisdate
                )
                while srctables:
                    if nxt(lasttable) > thisdate:
                        self.logger.info('Everything is finished, bye')
                        sys.exit(1)
                    conf_marker = self.process_table(srctables[0], **kwargs)
                    if not self.confirmation or conf_marker:
                        self.set_lasttable(format(srctables[0]))
                    lasttable = self.get_lasttable()
                    srctables = date_range(nxt(lasttable), thisdate)
            else:
                if not _to:
                    _to = thisdate
                srctables = date_range(_from, _to)
                for table in srctables:
                    self.process_table(table, **kwargs)

        if not args.nolock:
            self.unlock()
