from __future__ import unicode_literals

import sys

import nile
import scarab
import scarab.main

from nile.api.v1 import clusters, Record

if '/usr/share/nile/wheels' not in nile.config.WHEEL_PATH:
    nile.config.WHEEL_PATH.append('/usr/share/nile/wheels')
if 'scarab' not in nile.config.PACKAGES:
    nile.config.PACKAGES.append('scarab')
if 'rstr' not in nile.config.PACKAGES:
    nile.config.PACKAGES.append('rstr')


def parse(record):
    uid, reqid, text = record['text'].split('\t')
    parsed = parse_main_text(text)
    parsed.update({
            'uid': uid,
            'reqid': reqid,
            '_stbx': record['_stbx'],
            'iso_eventtime': record['iso_eventtime'],
            'source_uri': record['source_uri'],
            'subkey': record['subkey'],
            'tskv_format': record['tskv_format'],
            'unixtime': record['unixtime']
        })
    return parsed


def parse_main_text(text):
    import scarab.main
    parsed = {}
    try:
        data = scarab.main.deserialize_event_from_str(text, True)
    except Exception as error:
        sys.stderr.write('Failed to parse record\'s data\n. Error: {}'.format(error))
    else:
        if data.type == 'ATOM_FRONT_REQUEST_EVENT':
            parsed = parse_main_data_request(data)
        if data.type == 'ATOM_FRONT_ANSWER_EVENT':
            parsed = parse_main_data_answer(data)
        parsed['type'] = data.type
    return parsed


def parse_main_data_request(main_data):
    return parse_main_data_common(main_data)


def parse_main_data_answer(main_data):
    data = parse_main_data_common(main_data)
    data['answer'] = parse_answers(main_data.answer)
    return data


def parse_main_data_common(data):
    if data is not None:
        return {
            'client': data.client,
            'reqid': data.client_reqid,
            'login_hash': data.login_hash,
            'provider': data.provider,
            'request_id': getattr(data.request_id, 'value', None),
            'timestamp': data.timestamp,
            'version': data.version,
            'user_id': parse_user_id(data.user_id)
        }
    else:
        return {}


def parse_user_id(user_id):
    if user_id is not None:
        return {
            'device_id': getattr(user_id.device_id, 'value', None),
            'passport_uid': getattr(user_id.passport_uid, 'value', None),
            'platform_id': getattr(user_id.platform_id, 'value', None),
            'platform_id2': getattr(user_id.platform_id2, 'value', None),
            'robot_uid': getattr(user_id.robot_uid, 'value', None),
            'uuid': getattr(user_id.uuid, 'value', None),
            'yandex_fuid': getattr(user_id.yandex_fuid, 'value', None),
            'yandex_login': getattr(user_id.yandex_login, 'value', None),
            'yandex_uid': getattr(user_id.yandex_uid, 'value', None)
        }
    else:
        return None


def parse_answers(answers):
    if answers is not None:
        return [parse_answer(answer) for answer in answers]
    else:
        return None


def parse_answer(answer):
    if answer is not None:
        return {
            'aux_info': parse_aux_info(answer.aux_info),
            'collect_pool_mode': answer.collect_pool_mode,
            'docs': parse_docs(answer.docs),
            'docs_unranked': parse_docs(answer.docs_unranked),
            'error': answer.error,
            'message': answer.message,
            'name': answer.name,
            'rerank_success': answer.rerank_success,
            'sources': parse_sources(answer.sources),
            'warnings': parse_warnings(answer.warnings)
        }
    else:
        return None


def parse_sources(sources):
    return list(sources) if sources is not None else []


def parse_warnings(warnings):
    return list(warnings) if warnings is not None else []


def parse_aux_info(aux_info):
    if aux_info is not None:
        return {
            'props': getattr(aux_info.props, 'json', None),
            'candidate_infos': parse_candidate_infos(aux_info.candidate_infos),
        }
    else:
        return None


def parse_candidate_infos(infos):
    if infos is not None:
        return [parse_info(info) for info in infos]
    else:
        return None


def parse_docs(docs):
    if docs is not None:
        return [parse_info(doc) for doc in docs]
    else:
        return None


def parse_info(info):
    return {
        'banner_id': info.banner_id,
        'beta_params': parse_beta_params(info.beta_params),
        'boost_featuredump': parse_features(info.boost_featuredump),
        'featuredump': parse_features(info.featuredump),
        'initial_pos': info.initial_pos,
        'link': info.link,
        'score': info.score,
        'source_aux': getattr(info.source_aux, 'json', None),
        'weighted_score': info.weighted_score
    }


def parse_features(features):
    return list(features) if features is not None else None


def parse_beta_params(beta_params):
    if beta_params is not None:
        return {
            'alpha': beta_params.alpha,
            'beta': beta_params.beta
        }
    else:
        return None


def usage_example(date):
    def parser_mapper(records):
        for record in records:
            try:
                parsed = parse(record)
                yield Record(**parsed)
            except Exception as error:
                sys.stderr.write('Failed to parse record. Error {}\n'.format(error))

    import getpass

    username = getpass.getuser()

    cluster = clusters.Hahn(pool='search-research_{}'.format(username)).env(
        templates=dict(job_root='home/search-research/{}'.format(username)))

    job = cluster.job()

    job.table('//statbox/atomfront-answer-log/{}'.format(date))\
        .random(fraction=0.01)\
        .map(parser_mapper)\
        .put('$job_root/atomfront-answer-log-parsed_{}'.format(date))
    job.run()

if __name__ == '__main__':
    usage_example(date='2016-11-20')
