#!/usr/bin/python
# -*- coding: UTF-8 -*-

import json
import toloka
from datetime import date, timedelta
from collections import namedtuple
from time import sleep
from config import products as cfg


class TolokaHelper:
    """
    For Toloka API documentation reference
    https://tech.yandex.ru/toloka/api/doc/concepts/about-docpage/
    """
    token = 'AQAAAAAS64C8AACtpfjTQ5fpiU-Uq-P7mdazEhE'
    api = 'https://toloka.yandex.ru/api/v1'

    def __init__(self, project_descriptor, url_pattern, _date,
                 api=api, token=token):
        self.accepted_status = 'ACCEPTED'
        self.pools_limit = 300
        self.assignments_limit = 400
        self.date = _date
        self.project_descriptor = project_descriptor
        self.url_pattern = url_pattern
        self.toloka_stages = []
        self.tlk = toloka.Toloka(api=api, token=token)
        # import pdb; pdb.set_trace()
        self.all_pools = self.get_all_pools()
        self.projects = self.get_projects()
        self.pool_ids = self.get_pool_ids()

    # Helpers
    def get_all_pools(self):
        return [
            pool for pool in
            self.tlk.pools(
                limit=self.pools_limit,
                last_started_gte='{}T{}'.format(self.date, '00:00'),
                last_started_lte='{}T{}'.format(self.date, '23:59'),
                status='CLOSED'
            )['items']
        ]

    def get_projects(self):
        return [
            project['id']
            for project in self.tlk.projects(limit=300)['items']
            if self.project_descriptor in project['public_name']
        ]

    def get_pool_ids(self):
        '''Returns all pool_ids for self.projects'''
        return [
            pool['id']
            for pool in self.all_pools
            if pool['project_id'] in self.projects
        ]

    def get_assignments(self, pool_id):
        return self.tlk.assignments(pool_id, self.assignments_limit)['items']

    def get_stage_pools_daily(self):
        stage_pools = []

        def pool_stage(name):
            if '_' in name and name.split('_')[1].startswith('stage'):
                return int(name[-1])
            else:
                return 100500

        for pool in self.all_pools:
            if pool['id'] in self.pool_ids:
                status, created = pool['status'], pool['created']
                if self.date in created and status in ['CLOSED', 'COMPLETED']:
                    stage_pools.append((pool['id'],
                                        pool_stage(pool['private_name'])))
                print pool['id'], status, created
        stage_pools = sorted(stage_pools, key=lambda x: x[1])
        stage_pools = [pair[0] for pair in stage_pools]
        return stage_pools

    # Review assignments
    def version_ok(self, ver, kpi):
        # Deal with versions like '8.22'
        if '.' in kpi:
            if '.' not in ver:
                return False
            ver = ver.strip().split('.')
            kpi = kpi.split('.')
            for check, right in zip(ver, kpi):
                if not check.isdigit():
                    return False
                if int(check) > int(right):
                    return True
                elif int(check) == int(right):
                    continue
                else:
                    return False
            return True
        # Deal with versions like '35'
        else:
            if not ver.isdigit():
                return False
            if int(ver) < int(kpi):
                return False
            return True

    def check_solution_stage_0(self, right_answer, submitted):
        bad_for_reject = [
            task_suite['id'] for task_suite in
            submitted if not self.version_ok(
                ver=task_suite['solutions'][0]['output_values']['ver'],
                kpi=right_answer
            )
        ]
        good_for_accept = [
            task_suite['id'] for task_suite in
            submitted if self.version_ok(
                ver=task_suite['solutions'][0]['output_values']['ver'],
                kpi=right_answer
            )
        ]
        return bad_for_reject, good_for_accept

    def check_solution_other_stages(self, good_workers, submitted):
        bad_for_reject = [assignment['id'] for assignment in
                          submitted if assignment['user_id'] not in
                          good_workers]
        good_for_accept = [assignment['id'] for assignment in
                           submitted if assignment['user_id'] in
                           good_workers]
        return bad_for_reject, good_for_accept

    def get_stage0_good_workers(self, pools_daily):
        pool = pools_daily[0]
        pool_assignments = self.get_assignments(pool_id=pool)
        accepted = [assignment for assignment in
                    pool_assignments if assignment['status'] == 'ACCEPTED']
        good_toloker_ids = [assignment['user_id'] for assignment in accepted]
        return good_toloker_ids

    def autoreview_results(self, pools_daily, stage):
        pool_id = pools_daily[stage]
        pool_assignments = self.get_assignments(pool_id)
        submitted = [assignment for assignment in
                     pool_assignments if assignment['status'] == 'SUBMITTED']
        print 'Reviewing Stage: {}. Pool: {}'.format(stage, pool_id)
        for assignment in submitted:
            payload = json.dumps({'status': self.accepted_status})
            self.tlk.review_assignment(assignment['id'], payload)
        print '..done'

    def review_results(self, pools_daily, stage, right_answer=''):
        ReviewResult = namedtuple('review_result', ['bad', 'good'])
        pool_id = pools_daily[stage]
        pool_assignments = self.get_assignments(pool_id=pool_id)
        submitted = [assignment for assignment in
                     pool_assignments if assignment['status'] == 'SUBMITTED']
        if stage == 0 and right_answer:
            print 'Reviewing Stage: {}. Pool: {}'.format(stage, pool_id)
            bad_for_reject, good_for_accept = self.check_solution_stage_0(
                right_answer=right_answer,
                submitted=submitted
            )
        elif stage > 0:
            print 'Reviewing Stage: {}. Pool: {}'.format(stage, pool_id)
            good_workers = self.get_stage0_good_workers(pools_daily)
            bad_for_reject, good_for_accept = self.check_solution_other_stages(
                good_workers=good_workers,
                submitted=submitted
            )
        for task_suite in bad_for_reject:
            payload = json.dumps({"status": "REJECTED",
                                  "public_comment": "Bad version"})
            self.tlk.review_assignment(task_suite, payload)
        for task_suite in good_for_accept:
            payload = json.dumps({"status": self.accepted_status})
            self.tlk.review_assignment(task_suite, payload)
        return ReviewResult(bad_for_reject, good_for_accept)

    # Fetch results
    @staticmethod
    def get_id_from_url(url):
        for part in url.split('&'):
            if part.startswith('id='):
                return int(part.split('id=')[1].split('_')[0])

    def fetch_toloka_results(self, pools_daily, delay=False):
        print 'Fetch toloka results:'
        for idx, pool_id in enumerate(pools_daily):
            toloka_stage = {
                'id': idx,
                'indices': set(),
            }
            for assignment in self.get_assignments(pool_id=pool_id):
                if assignment['status'] != self.accepted_status:
                    continue
                url = assignment['tasks'][0]['input_values']['id']
                _id = self.get_id_from_url(url)
                toloka_stage['indices'].add(_id)
            self.toloka_stages.append(toloka_stage)
            print 'Stage: {}\tIndices count: {}'.format(
                toloka_stage['id'], len(toloka_stage['indices'])
            )
        if delay:
            self.toloka_stages = [
                stage for stage in self.toloka_stages
                if stage['id'] == 3
            ]

    # New assignments
    def clone_pool(self, original_pool_id, stage, delay_skill=False):
        original_pool = self.tlk.pool(original_pool_id)
        today_at_10pm = '{}{}'.format(date.today().isoformat(), 'T22:00:00')
        today_at_1010pm = '{}{}'.format(date.today().isoformat(), 'T22:10:00')
        today_at_1159pm = '{}{}'.format(date.today().isoformat(), 'T23:59:59')
        new_pool = {
            k: v for k, v in original_pool.iteritems()
            if k not in (
                'created',
                'id',
                'last_started',
                'last_stopped',
                'status'
            )
        }
        new_pool['private_name'] = '{}_stage{}'.format(date.today(), stage)
        zero_stage_pools = (cfg[product]['pool_zero'] for product in cfg)
        first_stage_pools = (cfg[product]['pool_one'] for product in cfg)
        second_stage_pools = (cfg[product]['pool_two'] for product in cfg)
        third_stage_pools = (cfg[product]['pool_three'] for product in cfg)
        if str(original_pool_id) in zero_stage_pools:
            new_pool['will_expire'] = today_at_10pm
        elif str(original_pool_id) in first_stage_pools:
            new_pool['will_expire'] = today_at_1010pm
        elif str(original_pool_id) in second_stage_pools:
            new_pool['will_expire'] = today_at_1159pm
        elif str(original_pool_id) in third_stage_pools:
            new_pool['will_expire'] = today_at_1159pm
        else:
            print 'Add original pools: pool_zero, pool_one, pool_two,\
                   pool_three to config'
            raise ImportWarning

        if delay_skill:
            new_pool['filter']['and'].append(
                {u'or': [
                    {u'category': u'skill',
                     u'key': delay_skill,
                     u'operator': u'EQ',
                     u'value': u'1'}
                ]})
        new_pool = self.tlk.create_pool(**new_pool)
        print 'Pool {} created'.format(new_pool['id'])
        return new_pool['id']

    def fill_data(self, pool_id, stage, product, tasks_count):
        today = date.today().isoformat()
        for _id in xrange(tasks_count):
            data = {
                'pool_id': pool_id,
                'tasks': [],
                'overlap': 1,
            }
            tasks = {
                'input_values': {
                    'id': self.url_pattern.format(
                        _id=str(_id).zfill(6),
                        stage=stage,
                        product=product,
                        date=today
                    )
                }
            }
            data['tasks'].append(tasks)
            try:
                self.tlk.upload_tasks(**data)
            except:
                sleep(1)
                self.tlk.upload_tasks(**data)
        print 'Product: {}\tStage: {}\tPool ID: {}\tFilled with data'.format(
            product, stage, pool_id
        )

    def start_new_pool(self, original_pool_id, product, stage, tasks_count,
                       delay_skill=False):
        new_pool_id = self.clone_pool(original_pool_id, stage,
                                      delay_skill=delay_skill)
        self.fill_data(pool_id=new_pool_id, stage=stage, product=product,
                       tasks_count=tasks_count)
        self.tlk.open_pool(new_pool_id)
        print 'Pool {} opened\n'.format(new_pool_id)

    def give_assignments(self, product, pool_zero, pool_one, pool_two,
                         tasks_count):
        self.start_new_pool(original_pool_id=pool_zero,
                            product=product,
                            stage=0,
                            tasks_count=tasks_count)
        sleep(300)
        self.start_new_pool(original_pool_id=pool_one,
                            product=product,
                            stage=1,
                            tasks_count=tasks_count)
        sleep(2 * 60 * 60)
        self.start_new_pool(original_pool_id=pool_two,
                            product=product,
                            stage=2,
                            tasks_count=tasks_count)
