#!/usr/bin/env python
# -*- coding: utf-8 -*-

import requests
import requests.packages.urllib3
import json
import os
from time import sleep

requests.packages.urllib3.disable_warnings()


class _TolokaRequest:
    def __init__(self, request, address, access_token,
                 data=None, params=None, headers=None):
        data = bytes(self._encode_data(data))
        params = self._encode_dict(params)
        headers = self._encode_dict(headers)
        headers = self._prepare_headers(headers, access_token)
        self.request = request(url=address, data=data,
                               headers=headers, params=params, verify=False)
        self.status_code = self.request.status_code
        self.text = self.request.text
        try:
            self.json = self.request.json()
        except ValueError:
            self.json = None
        if isinstance(self.json, dict) and self.json.get('success') is False:
            self.error = self.json.get('msg')
            self.success = False
        else:
            self.error = None
            self.success = self.status_code >= 200 and self.status_code < 300

    def __repr__(self):
        return "<_TolokaRequest code:{0}>".format(self.status_code)

    def __str__(self):
        if self.json is not None:
            return json.dumps(self.json)
        elif self.text:
            return self.text
        else:
            return "code:{0}".format(self.status_code)

    def _encode_data(self, data):
        if isinstance(data, dict) or isinstance(data, list):
            data = json.dumps(data)
        if isinstance(data, unicode):
            data = data.encode('utf-8')
        if isinstance(data, str):
            try:
                data = bytearray(data, 'utf-8')
            except UnicodeDecodeError:
                data = bytearray(data)
        return data

    def _encode_dict(self, headers):
        if isinstance(headers, dict):
            headers = dict((k.encode('ascii') if isinstance(k, unicode) else k,
                            v.encode('ascii') if isinstance(v, unicode) else v)
                           for k, v in headers.iteritems())
        return headers

    def _prepare_headers(self, headers, access_token):
        auth = {"Authorization": "OAuth {0}".format(access_token),
                "Content-Type": "application/json;charset=UTF-8"}
        if not headers:
            return auth
        else:
            headers.update(auth)
            return headers


class _TolokaRequestException(Exception):
    def __init__(self, request):
        message = """Toloka request has failed.
Code: {0}.
URL: {1}.
Error: {2}.
json: {3}.""".format(request.status_code,
                     request.request.url,
                     request.error,
                     json.dumps(request.json, sort_keys=True, indent=2))
        super(_TolokaRequestException, self).__init__(message)
        self.request = request


class TolokaAPI:
    def __init__(self, api_url, access_token, strict=False):
        self.api_url = api_url
        self.access_token = access_token
        self.strict_mode = strict

    def get(self, address, data=None, params=None):
        return self._request(requests.get, address, data, params)

    def post(self, address, data=None, params=None):
        return self._request(requests.post, address, data, params)

    def delete(self, address, data=None, params=None):
        return self._request(requests.delete, address, data, params)

    def put(self, address, data=None, params=None):
        return self._request(requests.put, address, data, params)

    def patch(self, address, data=None, params=None):
        return self._request(requests.patch, address, data, params)

    def _request(self, request, address, data=None, params=None):
        request = _TolokaRequest(
            request, self.api_url + address, self.access_token, data, params
        )
        if self.strict_mode and not request.success:
            raise _TolokaRequestException(request)
        return request


def read_first_line(path):
    if os.path.isfile(path):
        with open(path) as f:
            return f.readline().strip()


class Toloka:
    def __init__(self, api=None, token=None, strict=True):
        api = api or read_first_line("toloka.url") or 'https://toloka.yandex.com/api/external'
        token = token or read_first_line("toloka.token")
        self.api = TolokaAPI(api, token, strict=strict)

    # Used
    def assignments(self, poolid, limit, **kwargs):
        while True:
            try:
                return self.api.get("/assignments?pool_id={}&limit={}".format(
                    poolid, limit), params=kwargs).json
            except requests.exceptions.RequestException as e:
                print e
                sleep(2)

    def create_pool(self, **kwargs):
        while True:
            try:
                return self.api.post("/pools", data=kwargs).json
            except requests.exceptions.RequestException as e:
                print e
                sleep(2)

    def open_pool(self, poolId):
        while True:
            try:
                return self.api.post("/pools/{0}/open".format(poolId))
            except requests.exceptions.RequestException as e:
                print e
                sleep(2)

    def pool(self, poolId):
        while True:
            try:
                return self.api.get("/pools/{0}".format(poolId)).json
            except requests.exceptions.RequestException as e:
                print e
                sleep(2)

    def pools(self, limit, last_started_gte, last_started_lte, status):
        while True:
            try:
                return self.api.get(
                    "&".join([
                        "/pools?status={}",
                        "sort=created",
                        "limit={}",
                        "last_started_gte={}",
                        "last_started_lte={}"
                    ]).format(status, limit, last_started_gte,
                              last_started_lte)
                ).json
            except requests.exceptions.RequestException as e:
                print e
                sleep(2)

    def projects(self, limit):
        while True:
            try:
                return self.api.get(
                    "/projects/?sort=created&limit={}".format(limit)).json
            except requests.exceptions.RequestException as e:
                print e
                sleep(2)

    def review_assignment(self, task_suite_id, data):
        while True:
            try:
                return self.api.patch("/assignments/{}".format(task_suite_id), data=data).json
            except requests.exceptions.RequestException as e:
                print e
                sleep(2)

    def upload_tasks(self, **kwargs):
        while True:
            try:
                return self.api.post("/task-suites", data=kwargs).json
            except requests.exceptions.RequestException as e:
                print e
                sleep(2)

    def set_worker_skill(self, **kwargs):
        return self.api.put("/user-skills", data=kwargs)

    # Not used
    def project(self, projectId):
        return self.api.get("/projects/{0}".format(projectId)).json

    def alter_project(self, projectId, **kwargs):
        return self.api.put("/projects/{0}".format(projectId), data=kwargs).json

    def ban_worker(self, projectId, workerId, **kwargs):
        return self.api.patch("/projects/{0}/workers/{1}".format(projectId, workerId), data=kwargs).json

    def banned_workers(self, projectId):
        return self.api.get("/projects/{0}/workers?banned=true".format(projectId)).json

    def close_pool(self, poolId):
        return self.api.post("/pools/{0}/close".forma(poolId))

    def tasks(self, poolId):
        return self.api.get("/projects/pools/{0}/tasks".format(poolId)).json

    def task(self, taskId):
        return self.api.get("/projects/pools/tasks/{0}".format(taskId)).json

    def delete_task(self, taskId):
        return self.api.delete("/projects/pools/tasks/{0}".format(taskId)).json

    def workers(self, **kwargs):
        return self.api.get("/workers", params=kwargs).json

    def worker(self, workerId):
        return self.api.get("/workers/{0}".format(workerId)).json

    def alter_worker(self, workerId, **kwargs):
        return self.api.patch("/workers/{0}".format(workerId), data=kwargs).json

    def skills(self):
        return self.api.get("/skills").json

    def skill(self, skillId):
        return self.api.get("/skills/{0}".format(skillId)).json

    def create_skill(self, **kwargs):
        return self.api.post("/skills", data=kwargs).json

    def alter_skill(self, skillId, **kwargs):
        return self.api.patch("/skills/{0}".format(skillId), data=kwargs).json

    def delete_skill(self, skillId):
        return self.api.delete("/skills/{0}".format(skillId)).json

    def clear_worker_skill(self, workerId, skillId):
        return self.api.delete("/workers/{0}/skills/{1}".format(workerId, skillId)).json

    def workers_skills_csv(self):
        return self.api.get("/workers/skills.csv").text

    def assignments_csv(self, poolId):
        return self.api.get("/projects/pools/{0}/assignments.csv".format(poolId)).text

    def results(self, poolId, **kwargs):
        return self.api.get("/pools/{0}/results".format(poolId), params=kwargs).json

    def get_tasks(self, poolId):
        return self.api.get("/task-suites?pool_id={0}".format(poolId)).json
