# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os
import argparse
import datetime
import getpass

from yamon.configs.template_config import TemplateConfig
from yamon.monitorers.atom_distribution_alerts_monitorer import AtomDistributionAlertsMonitorer
from yamon.pushers.atom_distribution_alerts_pusher import AtomDistributionAlertsPusher
from yamon.providers.atom_distribution_alerts_slices_provider import AtomDistributionAlertsSlicesProvider
from yamon.cleaners.atom_distribution_alerts_cleaner import AtomDistributionAlertsCleaner
from yamon.managers.daily_monitoring_manager import DailyMonitoringManager
from yamon.utils.monitorings import get_monitoring_name, get_yamon_path
from yamon.utils.json import load_json


def main(config):
    config['user'] = config['user'] or getpass.getuser()
    config['monitorer']['pool'] = (config['monitorer'].get('pool') or
        config['monitorer'].get('pool_pattern', '{}').format(config['user']))
    for user, user_config in config['monitorer']['users'].items():
        for notifier_config in user_config['pusher']['notifiers']:
            if 'ya_stat_auth' in notifier_config:
                notifier_config['ya_stat_auth']['auth_headers'] = load_json(
                    notifier_config['ya_stat_auth']['ya_stat_token_path']
                )

    monitorers = [
        AtomDistributionAlertsMonitorer(
            date=config['date'],
            leap=config['leap'],
            pusher=AtomDistributionAlertsPusher(do_push=config['do_push']),
            provider=AtomDistributionAlertsSlicesProvider(),
            cleaner=AtomDistributionAlertsCleaner(),
            config=config['monitorer']
        )
    ]
    for monitorer in monitorers:
        monitorer.run()


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--from_date', type=str, default=None, help='First date to be processed')
    parser.add_argument('--to_date', type=str, default=None, help='Last date to be processed')
    parser.add_argument('--leap', type=int, default=7, help='Timespan in days between compared dates')
    parser.add_argument('--no_lock', action='store_true', help='Run monitoring even if other instance already in process')
    parser.add_argument('--with_today', action='store_true', help='Run monitoring for dates including today')
    parser.add_argument('--no_push', action='store_true', help='Do not send results anywhere')
    parser.add_argument('--user', type=str, default=None, help='User name in cluster pool')
    parser.add_argument('--force', action='store_true', help='Run monitoring even for requested dates no matter what')
    parser.add_argument(
        '--log_mode', type=str, default='default',
        choices=['default', 'document', 'console', 'email', 'verbose', 'debug'],
        help=(
              'default - log to file and send critical emails, '
              'document - log to file, '
              'debug - detailed log to console, '
              'console - log to console, '
              'email - send critical emails, '
              'verbose - detailed log to file and console and send critical emails'
        )
    )

    args = parser.parse_args()

    path = os.path.abspath(__file__)
    name = get_monitoring_name(path)
    yamon_path = get_yamon_path(path)
    config = TemplateConfig(
        source='{}/watchers/{}/config.json'.format(yamon_path, name),
        templates={
            '$YAMON_PATH': yamon_path,
            '$NAME': name,
            '$LOG_TYPE': name,
            '$LOG_NAME': '{}.log'.format(datetime.datetime.now().strftime('%Y-%m-%d_%H-%M-%S'))
        }
    )

    config['manager']['from_date'] = args.from_date
    config['manager']['to_date'] = args.to_date
    config['manager']['use_lock'] = not args.no_lock
    config['manager']['force'] = args.force
    config['manager']['with_today'] = args.with_today
    config['manager']['log_mode'] = args.log_mode
    config['monitoring']['do_push'] = not args.no_push
    config['monitoring']['leap'] = args.leap
    config['monitoring']['user'] = args.user

    manager = DailyMonitoringManager(name=name, config=config['manager'])
    manager.run(main, config['monitoring'])
