# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os
import argparse
import datetime
import getpass

from yamon.configs.template_config import TemplateConfig
from yamon.monitorers.atom_distribution_monitorer import AtomDistributionMonitorer
from yamon.pushers.atom_distribution_pusher import AtomDistributionPusher
from yamon.providers.atom_distribution_provider import AtomDistributionProvider
from yamon.providers.mobile_apps_name_provider import MobileAppsNameProvider
from yamon.providers.atom_context_provider import AtomContextProvider
from yamon.mergers.atom_distribution_merger import AtomDistributionMerger
from yamon.aggregators.atom_distribution_aggregator import AtomDistributionAggregator
from yamon.managers.daily_monitoring_manager import DailyMonitoringManager
from yamon.utils.monitorings import get_monitoring_name, get_yamon_path
from yamon.utils.tools import load_file, byteify
from yamon.utils.json import load_json


def main(config):
    config['user'] = config['user'] or getpass.getuser()
    config['monitorer']['pool'] = (config['monitorer'].get('pool') or
        config['monitorer'].get('pool_pattern', '{}').format(config['user']))
    namings_config = config['provider']['metrika_mobile_log']['namings']
    namings_config['ya_stat_provider']['ya_stat_auth']['auth_headers']['Authorization'] = load_file(
        namings_config['ya_stat_provider']['ya_stat_auth']['ya_stat_token_path']
    )
    namings_provider = MobileAppsNameProvider(namings_config)
    namings_config['api_keys'] = namings_provider.get('api_keys')
    namings_config['app_ids'] = namings_provider.get('app_ids')
    config['merger']['showid_parser']['context_provider']['arcanum_token'] = load_file(
        config['merger']['showid_parser']['context_provider']['arcanum_token_path']
    )
    config['merger']['showid_parser']['contexts'] = AtomContextProvider(
        config['merger']['showid_parser']['context_provider']
    ).get()
    config['merger']['testids_provider']['arcanum_token'] = load_file(
        config['merger']['testids_provider']['arcanum_token_path']
    )
    config['merger']['adata_parser'].update(
        load_json(config['merger']['adata_parser']['ivkey_path'])
    )
    config['pusher']['ya_stat_auth']['auth_headers'] = load_json(
        config['pusher']['ya_stat_auth']['ya_stat_token_path']
    )
    # We need this hack because some functions in qb2 and nile
    # can't handle unicode
    config = byteify(config)

    monitorers = [
        AtomDistributionMonitorer(
            date=config['date'],
            pusher=AtomDistributionPusher(config['pusher'], do_push=config['do_push']),
            provider=AtomDistributionProvider(config['provider']),
            merger=AtomDistributionMerger(config['merger']),
            aggregator=AtomDistributionAggregator(config['aggregator']),
            config=config['monitorer']
        )
    ]
    for monitorer in monitorers:
        monitorer.run()


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--from_date', type=str, default=None, help='First date to be processed')
    parser.add_argument('--to_date', type=str, default=None, help='Last date to be processed')
    parser.add_argument('--no_lock', action='store_true', help='Run monitoring even if other instance already in process')
    parser.add_argument('--with_today', action='store_true', help='Run monitoring for dates including today')
    parser.add_argument('--no_push', action='store_true', help='Do not send results anywhere')
    parser.add_argument('--user', type=str, default=None, help='User name in cluster pool')
    parser.add_argument('--force', action='store_true', help='Run monitoring even for requested dates no matter what')
    parser.add_argument(
        '--log_mode', type=str, default='default',
        choices=['default', 'document', 'console', 'email', 'verbose', 'debug'],
        help=(
              'default - log to file and send critical emails, '
              'document - log to file, '
              'debug - detailed log to console, '
              'console - log to console, '
              'email - send critical emails, '
              'verbose - detailed log to file and console and send critical emails'
        )
    )

    args = parser.parse_args()

    path = os.path.abspath(__file__)
    name = get_monitoring_name(path)
    yamon_path = get_yamon_path(path)
    config = TemplateConfig(
        source='{}/watchers/{}/config.json'.format(yamon_path, name),
        templates={
            '$YAMON_PATH': yamon_path,
            '$NAME': name,
            '$LOG_TYPE': name,
            '$LOG_NAME': '{}.log'.format(datetime.datetime.now().strftime('%Y-%m-%d_%H-%M-%S'))
        }
    )

    config['manager']['from_date'] = args.from_date
    config['manager']['to_date'] = args.to_date
    config['manager']['use_lock'] = not args.no_lock
    config['manager']['force'] = args.force
    config['manager']['with_today'] = args.with_today
    config['manager']['log_mode'] = args.log_mode
    config['monitoring']['do_push'] = not args.no_push
    config['monitoring']['user'] = args.user

    manager = DailyMonitoringManager(name=name, config=config['manager'])
    manager.run(main, config['monitoring'])
