# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os
import argparse
import datetime

from yamon.pushers.clickhouse_pusher import ClickhousePusher
from yamon.configs.template_config import TemplateConfig
from yamon.monitorers.clickhouse_atom_monitorer import ClickhouseAtomMonitorer
from yamon.managers.timeout_monitoring_manager import TimeoutMonitoringManager
from yamon.utils.monitorings import get_monitoring_name, get_yamon_path
from yamon.utils.json import load_json


def main(config):
    config['pusher']['clickhouse_auth'] = load_json(
        config['pusher']['clickhouse_auth_path']
    )
    monitorers = [
        ClickhouseAtomMonitorer(
            action=config['action'],
            pusher=ClickhousePusher(config['pusher'], config['do_push']),
            config=config['monitorer']
        )
    ]
    for monitorer in monitorers:
        monitorer.run()


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--action', type=str, default='optimize',
        choices=['build', 'rebuild', 'create', 'drop', 'optimize', 'init'],
        help=(
              'build - create and init, '
              'rebuild - drop and build',
              'create - create database without tables',
              'drop - drop database and its tables',
              'optimize - optimize tables for queries'
              'init - create tables in existing database'
        )
    )
    parser.add_argument('--no_push', action='store_true', help='Do not send results anywhere')
    parser.add_argument(
        '--log_mode', type=str, default='default',
        choices=['default', 'document', 'console', 'email', 'verbose', 'debug'],
        help=(
              'default - log to file and send critical emails, '
              'document - log to file, '
              'debug - detailed log to console, '
              'console - log to console, '
              'email - send critical emails, '
              'verbose - detailed log to file and console and send critical emails'
        )
    )
    parser.add_argument('--timeout', type=float, default=None, help='Maximum time allowed for monitoring execution')

    args = parser.parse_args()

    path = os.path.abspath(__file__)
    name = get_monitoring_name(path)
    yamon_path = get_yamon_path(path)
    config = TemplateConfig(
        source='{}/watchers/{}/config.json'.format(yamon_path, name),
        templates={
            '$YAMON_PATH': yamon_path,
            '$NAME': name,
            '$LOG_TYPE': name,
            '$LOG_NAME': '{}.log'.format(datetime.datetime.now().strftime('%Y-%m-%d_%H-%M-%S'))
        }
    )

    config['manager']['log_mode'] = args.log_mode
    config['manager']['timeout'] = args.timeout
    config['monitoring']['do_push'] = not args.no_push
    config['monitoring']['action'] = args.action

    manager = TimeoutMonitoringManager(name=name, config=config['manager'])
    manager.run(main, config['monitoring'])
