# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os
import argparse
import datetime

from yamon.holders.caching_json_holder import CachingJsonHolder
from yamon.pushers.integral_distribution_metric_pusher import IntegralDistributionMetricPusher
from yamon.providers.integral_distribution_metric_weights_provider import IntegralDistributionMetricWeightsProvider
from yamon.providers.atom_context_provider import AtomContextProvider
from yamon.providers.atom_collections_data_provider import AtomCollectionsDataProvider
from yamon.providers.caching_provider_decorator import CachingProviderDecorator
from yamon.providers.detailed_json_provider import DetailedJsonProvider
from yamon.aggregators.integral_distribution_metric_aggregator import IntegralDistributionMetricAggregator
from yamon.configs.template_config import TemplateConfig
from yamon.monitorers.integral_distribution_metric_monitorer import IntegralDistributionMetricMonitorer
from yamon.managers.timeout_monitoring_manager import TimeoutMonitoringManager
from yamon.utils.monitorings import get_monitoring_name, get_yamon_path
from yamon.utils.tools import load_file


def main(config):
    config['provider']['context_provider']['arcanum_token'] = load_file(
        config['provider']['context_provider']['arcanum_token_path']
    )
    config['provider']['contexts'] = AtomContextProvider(config['provider']['context_provider']).get()
    for sub_monitoring, sub_config in config['monitorer'].items():
        sub_config['monitorer']['collections_data_provider']['atom_admin_token'] = load_file(
            sub_config['monitorer']['collections_data_provider']['atom_admin_token_path']
        )
        sub_config['monitorer']['lists'] = AtomCollectionsDataProvider(
            sub_config['monitorer']['collections_data_provider']
        ).get()
    monitorers = [
        IntegralDistributionMetricMonitorer(
            pusher=IntegralDistributionMetricPusher(config['pusher'], config['do_push']),
            provider=CachingProviderDecorator(DetailedJsonProvider(config['provider'])),
            holder=CachingJsonHolder(config['holder']),
            weighter=IntegralDistributionMetricWeightsProvider(config['weighter']),
            aggregator=IntegralDistributionMetricAggregator(),
            config=config['monitorer']
        )
    ]
    for monitorer in monitorers:
        monitorer.run()


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--no_push', action='store_true', help='Do not send results anywhere')
    parser.add_argument(
        '--log_mode', type=str, default='default',
        choices=['default', 'document', 'console', 'email', 'verbose', 'debug'],
        help=(
              'default - log to file and send critical emails, '
              'document - log to file, '
              'debug - detailed log to console, '
              'console - log to console, '
              'email - send critical emails, '
              'verbose - detailed log to file and console and send critical emails'
        )
    )
    parser.add_argument('--timeout', type=float, default=None, help='Maximum time allowed for monitoring execution')

    args = parser.parse_args()

    path = os.path.abspath(__file__)
    name = get_monitoring_name(path)
    yamon_path = get_yamon_path(path)
    config = TemplateConfig(
        source='{}/watchers/{}/config.json'.format(yamon_path, name),
        templates={
            '$YAMON_PATH': yamon_path,
            '$NAME': name,
            '$LOG_TYPE': name,
            '$LOG_NAME': '{}.log'.format(datetime.datetime.now().strftime('%Y-%m-%d_%H-%M-%S'))
        }
    )

    config['manager']['log_mode'] = args.log_mode
    config['manager']['timeout'] = args.timeout
    config['monitoring']['do_push'] = not args.no_push

    manager = TimeoutMonitoringManager(name=name, config=config['manager'])
    manager.run(main, config['monitoring'])
