# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os
import argparse
import datetime

from yamon.pushers.post_pusher import PostPusher
from yamon.pushers.clickhouse_atom_pusher import ClickhouseAtomPusher
from yamon.holders.caching_json_holder import CachingJsonHolder
from yamon.aggregators.clickhouse_atom_aggregator import ClickhouseAtomAggregator
from yamon.mergers.rtmr_detailed_json_merger import RtmrDetailedJsonMerger
from yamon.cleaners.detailed_table_cleaner import DetailedTableCleaner
from yamon.configs.template_config import TemplateConfig
from yamon.providers.detailed_json_provider import DetailedJsonProvider
from yamon.providers.atom_context_provider import AtomContextProvider
from yamon.providers.atom_collections_data_provider import AtomCollectionsDataProvider
from yamon.providers.caching_provider_decorator import CachingProviderDecorator
from yamon.monitorers.atom_diff_monitorer import AtomDiffMonitorer
from yamon.monitorers.atom_abs_monitorer import AtomAbsMonitorer
from yamon.monitorers.clickhouse_atom_ya_stat_monitorer import ClickhouseAtomYaStatMonitorer
from yamon.managers.timeout_monitoring_manager import TimeoutMonitoringManager
from yamon.utils.monitorings import get_monitoring_name, get_yamon_path
from yamon.utils.tools import load_file
from yamon.utils.json import load_json


def main(config):
    config['provider']['context_provider']['arcanum_token'] = load_file(
        config['provider']['context_provider']['arcanum_token_path']
    )
    config['provider']['contexts'] = AtomContextProvider(config['provider']['context_provider']).get()
    config['notify']['ya_stat_auth']['auth_headers'] = load_json(
        config['notify']['ya_stat_auth']['ya_stat_token_path']
    )
    for kind in ['diffs', 'abs']:
        sub_monitoring = config[kind]
        sub_monitoring['pusher']['clickhouse_auth'] = load_json(
            sub_monitoring['pusher']['clickhouse_auth_path']
        )
        sub_monitoring['monitorer']['collections_data_provider']['atom_admin_token'] = load_file(
            sub_monitoring['monitorer']['collections_data_provider']['atom_admin_token_path']
        )
        sub_monitoring['monitorer']['lists'] = AtomCollectionsDataProvider(
            sub_monitoring['monitorer']['collections_data_provider']
        ).get()
    provider = CachingProviderDecorator(DetailedJsonProvider(config['provider']))
    holder = CachingJsonHolder(config['holder'])
    aggregator = ClickhouseAtomAggregator()
    merger = RtmrDetailedJsonMerger()
    cleaner = DetailedTableCleaner(config['cleaner'])
    monitorers = [
        AtomDiffMonitorer(
            pusher=ClickhouseAtomPusher(config['diffs']['pusher'], config['do_push']),
            provider=provider,
            holder=holder,
            aggregator=aggregator,
            merger=merger,
            cleaner=cleaner,
            config=config['diffs']['monitorer']
        ),
        AtomAbsMonitorer(
            pusher=ClickhouseAtomPusher(config['abs']['pusher'], config['do_push']),
            provider=provider,
            aggregator=aggregator,
            cleaner=cleaner,
            config=config['abs']['monitorer']
        ),
        ClickhouseAtomYaStatMonitorer(
            pusher=PostPusher(config['do_push']),
            config=config['notify']
        )
    ]
    for monitorer in monitorers:
        monitorer.run()


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--no_push', action='store_true', help='Do not send results anywhere')
    parser.add_argument(
        '--log_mode', type=str, default='default',
        choices=['default', 'document', 'console', 'email', 'verbose', 'debug'],
        help=(
              'default - log to file and send critical emails, '
              'document - log to file, '
              'debug - detailed log to console, '
              'console - log to console, '
              'email - send critical emails, '
              'verbose - detailed log to file and console and send critical emails'
        )
    )
    parser.add_argument('--timeout', type=float, default=None, help='Maximum time allowed for monitoring execution')

    args = parser.parse_args()

    path = os.path.abspath(__file__)
    name = get_monitoring_name(path)
    yamon_path = get_yamon_path(path)
    config = TemplateConfig(
        source='{}/watchers/{}/config.json'.format(yamon_path, name),
        templates={
            '$YAMON_PATH': yamon_path,
            '$NAME': name,
            '$LOG_TYPE': name,
            '$LOG_NAME': '{}.log'.format(datetime.datetime.now().strftime('%Y-%m-%d_%H-%M-%S'))
        }
    )

    config['manager']['log_mode'] = args.log_mode
    config['manager']['timeout'] = args.timeout
    config['monitoring']['do_push'] = not args.no_push

    manager = TimeoutMonitoringManager(name=name, config=config['manager'])
    manager.run(main, config['monitoring'])
