# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import numpy
import pandas
import logging

from ..utils.time import get_now_ts


class AtomSlicesAggregator(object):
    def __init__(self, config):
        super(AtomSlicesAggregator, self).__init__()
        self.logger = logging.getLogger(__name__)
        self.config = config
        # CONTEXT + LIST = DIMENSIONS
        # DIMENSIONS + EVENT + COUNT + UPDATE_TS = COLUMNS
        self.context = [
            'host', 'product', 'region', 'referer',
            'client', 'subclient', 'browser', 'os'
        ]
        self.dimensions_routines = {
            'dimensions': self.__aggregate_dimensions,
            'allextensions': self.__aggregate_allextensions,
            'search_entity': self.__aggregate_search_entity
        }
        self.events_routines = {
            'events': self.__aggregate_events,
            'install_to_real_click': self.__aggregate_install_to_real_click,
            'event_to_show': self.__aggregate_event_to_show,
            'any_install': self.__aggregate_any_install
        }

    def aggregate(self, data):
        clean_data = self.__uniformize(data)
        self.logger.info('Aggregating data on slices')
        return {
            self.__make_signature(slice_info['dimensions']): self.__aggregate_slice(clean_data, slice_info)
            for slice_info in self.config['slices']
        }

    def __make_signature(self, dimensions):
        return tuple(
            (dim, True if dim in dimensions else False)
            for dim in ['list'] + self.context
        )

    def __aggregate_slice(self, data, slice_info):
        dimensions = slice_info['dimensions']
        self.logger.info('Aggregating data on dimensions {}'.format(dimensions))
        if 'list' not in dimensions:
            data = data.drop_duplicates(subset=self.context + ['bannerid', 'event'])
        aggregated = pandas.concat([
            self.dimensions_routines[routine](data, slice_info)
            for routine in slice_info['agg_spec']['dimensions']
        ])
        enriched = pandas.concat([
            self.events_routines[routine](aggregated, slice_info)
            for routine in slice_info['agg_spec']['events']
        ])
        enriched['update_ts'] = get_now_ts()
        return enriched

    def __uniformize(self, data):
        self.logger.info('Ensure that counts and timestamps from different lists are the same')
        banner_lists = data[['bannerid', 'list']].drop_duplicates()
        uniformized = data.groupby(self.context + ['bannerid', 'event'], as_index=False)\
            .aggregate({
                'count': numpy.max,
                'update_ts': numpy.max
            })
        return uniformized.merge(
                banner_lists,
                how='right',
                on='bannerid',
                sort=False
            )

    def __aggregate_dimensions(self, data, slice_info):
        return self.__reduce_to_columns(data, slice_info['dimensions'] + ['event'])

    def __aggregate_allextensions(self, data, slice_info):
        self.logger.info('Adding allextensions')
        allextensions = {'set', 'home', 'sethome', 'vb', 'vbch'}
        addon = data[data['product'].isin(allextensions)].copy()
        addon['product'] = 'allextensions'
        return self.__reduce_to_columns(addon, slice_info['dimensions'] + ['event'])

    def __aggregate_search_entity(self, data, slice_info):
        self.logger.info('Adding search_entites')
        mobile_lists = {
            'smart_banner', 'smart_banner_daas',
            'mobilefooter', 'promoliba'
        }
        search_products = {
            'browser', 'search', 'browser_zombie',
            'shortcut', 'default_search_mobilesafari_ios'
        }
        addon = data[data['list'].isin(mobile_lists)].copy()
        addon['product'] = addon['product'].apply(
            lambda product: 'search_entity' if product in search_products else 'other'
        )
        return self.__reduce_to_columns(addon, slice_info['dimensions'] + ['event'])

    def __aggregate_events(self, data, slice_info):
        return data.copy()

    def __aggregate_any_install(self, data, slice_info):
        self.logger.info('Aggregating any_install')
        addon = data[data['event'].str.contains('install|INSTALL')].copy()
        addon['event'] = 'any_install'
        return self.__reduce_to_columns(addon, slice_info['dimensions'] + ['event'])

    def __aggregate_event_to_show(self, data, slice_info):
        self.logger.info('Aggregating event_to_show')
        noshow = data[data['event'] != 'show']
        shows = data.loc[data['event'] == 'show', slice_info['dimensions'] + ['count']]\
            .rename(columns={'count': 'show_count'})
        shows = shows[shows['show_count'] != 0]
        addon = noshow.merge(shows, how='left', on=slice_info['dimensions'])
        addon['count'] = addon['count'] / addon['show_count']
        addon['event'] = addon['event'].apply(
            lambda event: '{}_to_show'.format(event)
        )
        return addon.dropna().drop('show_count', axis=1)

    def __aggregate_install_to_real_click(self, data, slice_info):
        self.logger.info('Aggregating install_to_real_click')
        real_clicks = data.loc[data['event'] == 'real_click', slice_info['dimensions'] + ['count']]\
            .rename(columns={'count': 'real_click_count'})
        real_clicks = real_clicks[real_clicks['real_click_count'] != 0]
        installs = data[data['event'].str.contains('install|INSTALL')]
        addon = installs.merge(real_clicks, how='left', on=slice_info['dimensions'])
        addon['count'] = addon['count'] / addon['real_click_count']
        addon['event'] = addon['event'].apply(
            lambda event: '{}_to_real_click'.format(event)
        )
        return addon.dropna().drop('real_click_count', axis=1)

    def __reduce_to_columns(self, data, columns):
        return data.groupby(columns, as_index=False)\
            .aggregate({
                'count': numpy.sum,
                'update_ts': numpy.max
            })
