# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging
import pandas
import collections

from ..utils.time import get_now_ts


class IntegralDistributionMetricAggregator(object):
    def __init__(self):
        super(IntegralDistributionMetricAggregator, self).__init__()
        self.logger = logging.getLogger(__name__)

    def aggregate(self, data):
        self.logger.info('Aggregating stats by {}'.format(self.__class__.__name__))
        stats = data['stats']
        weights = self.__prepare_weights(data['weights'])
        weighted_stats = self.__weigh_stats(stats, weights)
        integral_stats = self.__integrate_stats(weighted_stats)
        total = self.__integrate_total(integral_stats)
        return {
            'weights': weights,
            'stats': stats,
            'weighted_stats': weighted_stats,
            'integral_stats': integral_stats,
            'total': total
        }

    def __prepare_weights(self, weights):
        self.logger.info('Propare weights')
        return {key: table.assign(update_ts=get_now_ts()) for key, table in weights.items()}

    def __weigh_stats(self, stats, weights):
        self.logger.info('Scaling stats')
        return {
            'installs': self.__weigh(stats['installs'], weights['installs'], on='product'),
            'shows': self.__weigh(stats['shows'], weights['shows'], on='host'),
            'closes': self.__weigh(stats['closes'], weights['closes'], on='host')
        }

    def __weigh(self, stats, weights, on):
        self.logger.info('Scaling stats for {}'.format(on))
        weighted_stats = stats.merge(weights.drop('update_ts', axis=1), how='left', on=on)
        weighted_stats['weight'].fillna(0, inplace=True)
        weighted_stats['count'] *= weighted_stats['weight']
        weighted_stats.drop('weight', inplace=True, axis=1)
        return weighted_stats

    def __integrate_stats(self, stats):
        self.logger.info('Integrating stats')
        return {
            'installs': self.__integrate(stats['installs'], on='product'),
            'shows': self.__integrate(stats['shows'], on='host'),
            'closes': self.__integrate(stats['closes'], on='host')
        }

    def __integrate(self, stats, on):
        self.logger.info('Integrating stats for {}'.format(on))
        return stats.groupby(lambda index: 'all')\
            .aggregate({
                'count': 'sum',
                'update_ts': 'max',
                on: lambda values: '{}s'.format(on),
                'event': lambda values: collections.Counter(values).most_common(1)[0][0]})\
            .reset_index(drop=True)

    def __integrate_total(self, stats):
        channels = ['shows', 'closes', 'installs']
        return pandas.DataFrame([{
            'count': sum(stats[channel]['count'].sum() for channel in channels),
            'update_ts': max(stats[channel]['update_ts'].sum() for channel in channels)
        }])
