# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging
import requests


class RequesterBase(object):
    def __init__(self, **settings):
        super(RequesterBase, self).__init__()
        self.logger = logging.getLogger(__name__)
        self.config = {
            'max_retries': 4,
            'backoff_factor': 0.1,
            'preficies': ('http://', 'https://'),
            'encoding': 'utf8',
            'errors': 'replace'
        }
        self.config.update(settings)
        self.session = self.__class__._make_session(**self.config)

    def _make_request(self, method, url, *args, **kwargs):
        try:
            self.logger.info('Sending {}  request to {}'.format(method, url))
            return getattr(self.session, method.lower())(url, *args, **kwargs)
        except requests.exceptions.RequestException as error:
            self.logger.critical('Failed to send {} request. Error {}'.format(method, error))

    def _process_response(self, response):
        try:
            if response is not None:
                response.raise_for_status()
                return response.content.decode(self.config['encoding'], errors=self.config['errors'])
        except requests.exceptions.RequestException as error:
            explanation = response.content.decode(self.config['encoding'], errors=self.config['errors'])
            raise requests.exceptions.RequestException('{}: {}'.format(error, explanation))

    @classmethod
    def _make_session(cls, **settings):
        session = requests.Session()
        retries = requests.packages.urllib3.util.retry.Retry(
            total=settings['max_retries'],
            backoff_factor=settings['backoff_factor']
        )
        for prefix in settings['preficies']:
            session.mount(prefix, requests.adapters.HTTPAdapter(max_retries=retries))
        return session


class Requester(RequesterBase):
    def __init__(self, **settings):
        super(Requester, self).__init__(**settings)
        self.logger = logging.getLogger(__name__)

    def get(self, url, **kwargs):
        return self.request('GET', url, **kwargs)

    def post(self, url, data=None, json=None, **kwargs):
        return self.request('POST', url, data=data, json=json, **kwargs)

    def request(self, method, url, *args, **kwargs):
        return self._process_response(
            self._make_request(method, url, *args, **kwargs)
        )
