# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os
import copy
import logging
import traceback

from .monitoring_manager import MonitoringManager
from ..utils.time import parse_date, safe_parse_date, next_date, get_today, get_yesterday, from_ordinal
from ..utils.json import load_json, save_json


class DailyMonitoringStateHolder(object):
    def __init__(self, state_path):
        super(DailyMonitoringStateHolder, self).__init__()
        self.state_path = state_path
        self.logger = logging.getLogger(__name__)
        if not os.path.exists(self.state_path):
            self.logger.info('File {} doesn\'t exists. Creating a new one'.format(self.state_path))
            self.__save_state({'exec_status': 'free', 'last_processed_date': None})

    def set_exec_status(self, exec_status):
        self.logger.info('Setting exec_status {}'.format(exec_status))
        state = self.__load_state()
        state['exec_status'] = exec_status
        self.__save_state(state)
        self.logger.info('Status set')

    def get_exec_status(self):
        self.logger.info('Reading exec_status')
        state = self.__load_state()
        status = state['exec_status']
        self.logger.info('Status {} read'.format(status))
        return status

    def set_last_processed_date(self, date):
        self.logger.info('Setting last processed date to {}'.format(date))
        state = self.__load_state()
        state['last_processed_date'] = date
        self.__save_state(state)
        self.logger.info('Date set')

    def get_last_processed_date(self):
        self.logger.info('Reading last processed date')
        state = self.__load_state()
        date = state['last_processed_date']
        self.logger.info('Date {} read'.format(date))
        return date

    def __load_state(self):
        try:
            return load_json(self.state_path)
        except Exception:
            self.logger.critical('Can\'t load state!')
            raise

    def __save_state(self, state):
        try:
            save_json(self.state_path, state)
        except Exception:
            self.logger.critical('Can\'t save state!')
            raise


class DailyMonitoringManager(MonitoringManager):
    def __init__(self, name, config):
        super(DailyMonitoringManager, self).__init__(name, config)
        self.logger = logging.getLogger(__name__)
        self.state_holder = DailyMonitoringStateHolder(self.config['holder']['state_path'])

    def _run_internal(self, function, config):
        if self.config['use_lock']:
            if not self.__lock():
                return
        try:
            dates = self.__get_process_dates()
            self.logger.info('There are {} dates to process: {}'.format(len(dates), dates))
            for date in dates:
                self.__process_date(date, function, config)
                self.__update_last_processed_date(date)
        except (Exception, KeyboardInterrupt) as error:
            self.logger.critical(
                'Monitoring: {}\nAn unexpected error occured: {}\n{}'.format(
                    self.name,
                    error,
                    getattr(error, 'traceback', traceback.format_exc()))
            )
        if self.config['use_lock']:
            if not self.__unlock():
                return

    def __process_date(self, date, function, config):
        try:
            conf = copy.deepcopy(config)
            conf['date'] = date
            function(conf)
        except Exception as error:
            self.logger.critical('Failed to process date {}. Reason: {}'.format(date, error))
            raise

    def __update_last_processed_date(self, date):
        last_date = self.state_holder.get_last_processed_date()
        if last_date is not None and parse_date(date) <= parse_date(last_date):
            self.logger.warning(
                'Later date {} was already processed. '
                'I will not change the last processed date'.format(last_date)
            )
        else:
            self.state_holder.set_last_processed_date(date)

    def __lock(self):
        self.logger.info('Locking process')
        if self.state_holder.get_exec_status() == 'free':
            self.state_holder.set_exec_status('locked')
            self.logger.info('Locking completed')
            return True
        else:
            self.logger.critical(
                'Process is already locked! '
                'Try again later or change locked to free in state.json'
            )
            return False

    def __unlock(self):
        self.logger.info('Unlocking process')
        if self.state_holder.get_exec_status() == 'locked':
            self.state_holder.set_exec_status('free')
            self.logger.info('Unlocking completed')
            return True
        else:
            self.logger.critical('Process was already unlocked. Unexpected state!')
            return False

    def __get_process_dates(self):
        from_date = safe_parse_date(self.config['from_date'])
        to_date = safe_parse_date(self.config['to_date'])
        if (from_date is not None and to_date is not None and from_date > to_date):
            from_date, to_date = to_date, from_date
        start = self.__get_start_date(from_date).toordinal()
        finish = self.__get_finish_date(to_date).toordinal()
        dates = [
            format(from_ordinal(ordinal))
            for ordinal in range(start, finish + 1)
        ]
        return dates

    def __get_start_date(self, from_date):
        start_date = self.__get_pending_date()
        if from_date is not None:
            if self.config['force']:
                start_date = from_date
            else:
                start_date = max(start_date, from_date)
        return start_date

    def __get_finish_date(self, to_date):
        finish_date = self.__get_last_date()
        if to_date is not None:
            if self.config['force']:
                finish_date = to_date
            else:
                finish_date = min(finish_date, to_date)
        return finish_date

    def __get_pending_date(self):
        date = self.state_holder.get_last_processed_date()
        if date is None:
            return self.__get_last_date()
        return next_date(safe_parse_date(date))

    def __get_last_date(self):
        return get_today() if self.config['with_today'] else get_yesterday()
