# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os
import datetime


class DiskTools(object):
    @classmethod
    def file_systems(cls):
        phydevs = []
        with open("/proc/filesystems", "r") as filesystems_file:
            for line in filesystems_file:
                if not line.startswith("nodev"):
                    phydevs.append(line.strip())
        return phydevs

    @classmethod
    def partitions(cls, all=False):
        """
        Return all mountd partitions as a nameduple.
        If all == False return phyisical partitions only.
        """
        phydevs = cls.file_systems()
        partitions_info = []
        with open('/etc/mtab', "r") as partitions_file:
            for line in partitions_file:
                if not all and line.startswith('none'):
                    continue
                fields = line.split()
                device = fields[0]
                mountpoint = fields[1]
                fstype = fields[2]
                if not all and fstype not in phydevs:
                    continue
                if device == 'none':
                    device = ''
                partitions_info.append({
                    'device': device,
                    'mountpoint': mountpoint,
                    'fstype': fstype
                })
        return partitions_info

    @classmethod
    def usage(cls, path):
        """
        Return disk usage associated with path.
        """
        st = os.statvfs(path)
        free = (st.f_bavail * st.f_frsize)
        total = (st.f_blocks * st.f_frsize)
        used = (st.f_blocks - st.f_bfree) * st.f_frsize
        try:
            percent = (float(used) / total) * 100
        except ZeroDivisionError:
            percent = 0
        # NB: the percentage is -5% than what shown by df due to
        # reserved blocks that we are currently not considering:
        # http://goo.gl/sWGbH
        return {'total': total, 'used': used, 'free': free, 'percent': round(percent, 1)}

    @classmethod
    def partitions_usage(cls):
        return [(part, cls.usage(part['mountpoint'])) for part in cls.partitions()]


class LogsCleaner(object):
    def __init__(self, config):
        super(LogsCleaner, self).__init__()
        self.config = config

    def run(self, where):
        directories = [where] if isinstance(where, basestring) else where
        for directory in directories:
            if os.path.exists(directory):
                for log_path in self.__find_waste_logs(directory):
                    os.remove(log_path)

    def __find_waste_logs(self, directory):
        now = datetime.datetime.now()
        max_age = datetime.timedelta(days=self.config['max_days'])
        max_usage = self.config['max_usage']
        usage = DiskTools.usage('/')['percent'] / 100.
        to_remove = []
        for root, dirs, files in os.walk(directory):
            for file_name in files:
                if file_name.endswith(self.config['extension']):
                    file_path = '{}/{}'.format(root, file_name)
                    creation_ts = int(os.path.getctime(file_path))
                    age = now - datetime.datetime.fromtimestamp(creation_ts)
                    if age > max_age or usage > max_usage:
                        to_remove.append(file_path)
        return to_remove
