# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os
import copy
import datetime
import logging
import logging.config
import logging.handlers
import traceback

from .logs_cleaner import LogsCleaner


class MonitoringManager(object):
    def __init__(self, name, config):
        super(MonitoringManager, self).__init__()
        self.name = name
        self.config = self.__build_manager_config(config)

    def run(self, function, config):
        self.__prepare_logs()
        logging.config.dictConfig(self.config['logger']['settings'])
        self.logger = logging.getLogger(__name__)
        start_time = datetime.datetime.now()
        self.logger.info('Monitoring started at: {}'.format(start_time))
        self._run_internal(function, config)
        finish_time = datetime.datetime.now()
        self.logger.info('Monitoring finished  at: {}'.format(finish_time))
        self.logger.info('Total time: {}'.format(finish_time - start_time))

    def _run_internal(self, function, config):
        try:
            function(config)
        except Exception as error:
            self.logger.critical(
                'Monitoring: {}\nAn unexpected error occured: {}\n{}'.format(
                    self.name,
                    error,
                    getattr(error, 'traceback', traceback.format_exc())
                )
            )

    def __prepare_logs(self):
        logs_directory = self.config['logger']['logs_directory']
        if not os.path.exists(logs_directory):
            os.makedirs(logs_directory)
        cleaner = LogsCleaner(self.config['cleaner'])
        cleaner.run(logs_directory)

    def __build_manager_config(self, config):
        manager_config = copy.deepcopy(config)
        logs_directory = manager_config['logger']['logs_directory']
        log_name = manager_config['logger']['settings']['handlers']['document']['filename']
        manager_config['logger']['settings']['handlers']['document']['filename'] = '{}/{}'.format(
            logs_directory,
            log_name
        )
        strategies = self.__get_logging_strategies()
        if manager_config['log_mode'] in strategies:
            manager_config['logger']['settings']['loggers']['yamon'] = strategies[manager_config['log_mode']]
        return manager_config

    def __get_logging_strategies(self):
        return {
            'default': {
                'level': 'INFO',
                'handlers': ['document', 'email']
            },
            'document': {
                'level': 'INFO',
                'handlers': ['document']
            },
            'debug': {
                'level': 'DEBUG',
                'handlers': ['console']
            },
            'console': {
                'level': 'INFO',
                'handlers': ['console']
            },
            'email': {
                'level': 'CRITICAL',
                'handlers': ['email']
            },
            'verbose': {
                'level': 'DEBUG',
                'handlers': ['console', 'document', 'email']
            }
        }
