# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

from nile.api.v1 import (
    Record,
    extractors as ne,
    filters as nf
)
from qb2.api.v1 import (
    extractors as qe,
    filters as qf,
    resources as qr
)

from ..parsers.showid_parser import ShowidParser
from ..parsers.adata_parser import AdataParser
from ..providers.ab_testids_provider import AbTestidsProvider
from ..providers.arcadia_testids_provider import ArcadiaTestidsProvider
from ..utils.json import decode_json


class MergeMapper(object):
    def __init__(self, config):
        super(MergeMapper, self).__init__()
        self.config = config
        self.showid_parser = ShowidParser(config['showid_parser']['contexts'])
        self.adata_parser = AdataParser(self.config['adata_parser'])
        self.white_list_testids = set(
            ArcadiaTestidsProvider(self.config['testids_provider']).get() +
            AbTestidsProvider(self.config['ab_testids_provider']).get()
        )

    def __call__(self, records):
        for record in records:
            context = record.to_dict()
            self.__add_from_showid(context)
            self.__add_normal_testids(context)
            self.__add_past(context)
            yield Record(
                key=context['key'],
                reqid=context['reqid'],
                showid=context['showid'],
                client=context['client'],
                subclient=context['subclient'],
                unixtime=context['unixtime'],
                distr_obj=context['distr_obj'],
                eventtype=context['eventtype'],
                region=context['region'],
                bannerid=context['bannerid'],
                score=context['score'],
                device=context['device'],
                device_id=context['device_id'],
                os=context['os'],
                browser=context['browser'],
                referer=context['referer'],
                service=context['service'],
                product=context['product'],
                yandexuid=context['yandexuid'],
                uuid=context['uuid'],
                testids=context['normal_testids'],  # normal_testids
                type=context['type'],
                origin=context['origin'],
                past=context['past']
            )

    def __add_from_showid(self, context):
        showid = context['showid']
        context['client'] = self.showid_parser.get_client(showid)
        context['subclient'] = self.showid_parser.get_subclient(showid)
        context['browser'] = self.showid_parser.get_browser(showid)
        context['os'] = context['os'] or self.showid_parser.get_os(showid)
        if context['distr_obj']:
            if 'promolib' in context['distr_obj']:
                context['client'] = 'promolib'
                context['subclient'] = ''

    def __add_normal_testids(self, context):
        try:
            testids = decode_json(self.adata_parser.parse(context['adata']))
            testids = set(map(format, testids['test-ids']))
        except Exception:
            testids = set()
        if context['testids'] is not None:
            testids |= set(context['testids'])
        context['normal_testids'] = sorted(testids & self.white_list_testids)

    def __add_past(self, context):
        if context['past'] is None:
            context['past'] = 0


class AtomDistributionMerger(object):
    def __init__(self, config):
        super(AtomDistributionMerger, self).__init__()
        self.config = config
        self.logger = logging.getLogger(__name__)

    def merge(self, extracts):
        self.logger.info('Mering extracts {}'.format(extracts.keys()))
        self.__check_availability(extracts)
        logs_compound = self.__make_logs_compound(extracts)
        ios_compound = self.__make_ios_compound(logs_compound, extracts)
        compound = logs_compound.concat(ios_compound)
        if extracts['atom_cube_past'] is not None:
            compound = compound.concat(extracts['atom_cube_past'])
        return self.__make_composite(compound)

    def __make_composite(self, compound):
        self.logger.info('Building composite')

        def get_region(geobase, lr):
            try:
                description = format(geobase.region_by_id(lr))
                return description.split('(')[1].split(' ')[0].lower()
            except Exception:
                return None

        return compound.qb2(
            log='generic-yson-log',
            fields=[
                qe.log_fields(
                    'key', 'showid', 'reqid', 'distr_obj', 'eventtype', 'bannerid',
                    'device', 'device_id', 'service', 'os', 'referer', 'product', 'yandexuid',
                    'uuid', 'adata', 'testids', 'type', 'origin', 'past'
                ),
                qe.integer_log_fields('unixtime', 'score', 'lr'),
                qe.custom('region', get_region, qr.resource('Geobase'), 'lr')
            ],
            filters=[
                qf.defined('key')
            ],
            intensity='ultra_cpu'
        ).map(
            MergeMapper(self.config),
            intensity='ultra_cpu'
        )

    def __check_availability(self, extracts):
        self.logger.info('Checking if all necessary logs are available')
        names = [
            'redir_log', 'metrika_mobile_log',
            'export_access_log', 'mobile_tracking_log',
            'id_to_generalized_id', 'bs_watch_log'
        ]
        for name in names:
            if extracts[name] is None:
                raise RuntimeError('Log {} was not extracted! Probably it doesn\'t exist'.format(name))

    def __make_logs_compound(self, extracts):
        self.logger.info('Building logs compound')
        return extracts['redir_log'].concat(
            extracts['metrika_mobile_log'],
            extracts['export_access_log'],
            extracts['mobile_tracking_log']
        )

    def __make_ios_compound(self, logs_compound, extracts):
        self.logger.info('Building ios compound')
        ios_clicks = logs_compound.filter(
            nf.and_(
                nf.equals('eventtype', 'click'),
                nf.equals('product', 'default_search_mobilesafari_ios'),
            )
        )
        ids = extracts['id_to_generalized_id']
        visits = extracts['bs_watch_log']
        user_devices = ids.join(visits, by='yandexuid', type='inner')
        serp_ios_clicks = ios_clicks.join(visits, by='yandexuid', type='inner')
        promoliba_ios_clicks = ios_clicks.join(user_devices, by='device_id', type='inner')
        ios_compund = serp_ios_clicks.concat(promoliba_ios_clicks)
        return ios_compund.project(ne.all(), origin=ne.const('ios_compound'))
