# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import pandas
import logging

from .atom_abs_monitorer_base import AtomAbsMonitorerBase


class AtomAbsMonitorer(AtomAbsMonitorerBase):
    def __init__(self, pusher, provider, cleaner, aggregator, config):
        super(AtomAbsMonitorer, self).__init__(pusher, provider, aggregator)
        self.logger = logging.getLogger(__name__)
        self.cleaner = cleaner
        self.config = config

    def run(self):
        keys = self.config['lists']
        self.logger.info('{}: GATHERING STATS ON EVENTS COUNT ABSOLUTES.'
                         .format(self.__class__.__name__))
        self.logger.info('All candidates lists: {}.'.format(keys))
        self.logger.info('Total candidates lists: {}.'.format(len(keys)))
        urls = {
            key: self.config['lists_json_url'].format(key)
            for key in keys
        }
        self._run_internal(urls=urls)
        self.pusher.commit()
        self.logger.info('STATS ON EVENTS COUNT ABSOLUTES ARE SENT.')

    def _collect(self, stats):
        data = self.cleaner.clean(self._collect_internal(stats))
        if self.logger.isEnabledFor(logging.DEBUG):
            with pandas.option_context('display.max_rows', 100, 'display.width', 400):
                self.logger.debug('Collected absolutes:\n{}'.format(data))
        return data

    def _collect_internal(self, stats):
        data = []
        for key in stats:
            self.logger.info('Processing key: {}'.format(key))
            key_ts = stats[key]['update_ts']
            for bannerid, candidate in stats[key]['statistic']['bannerids'].items():
                self.logger.debug('Processing bannerid {}'.format(bannerid))
                banner_update_ts = candidate['banner_update_ts']
                host = candidate['host']
                product = candidate['product']
                for bin, counts in candidate['bins'].items():
                    region, referer, subclient, client, browser, os = bin.split(',')
                    for event, count in counts.items():
                        data.append({
                            'list': key,
                            'host': host,
                            'product': product,
                            'bannerid': bannerid,
                            'event': event,
                            'region': region,
                            'referer': referer,
                            'client': client,
                            'subclient': subclient,
                            'browser': browser,
                            'os': os,
                            'count': count,
                            'update_ts': key_ts,
                            'banner_update_ts': banner_update_ts
                        })
        return pandas.DataFrame(
            data,
            columns=['list', 'host', 'product', 'bannerid', 'event',
                     'region', 'referer', 'client', 'subclient', 'browser', 'os',
                     'count', 'update_ts', 'banner_update_ts']
        )
