# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import pandas
import logging

from .atom_diff_monitorer_base import AtomDiffMonitorerBase


class AtomDiffMonitorer(AtomDiffMonitorerBase):
    def __init__(self, pusher, provider, holder, merger, cleaner, aggregator, config):
        super(AtomDiffMonitorer, self).__init__(pusher, provider, holder, merger, aggregator)
        self.logger = logging.getLogger(__name__)
        self.cleaner = cleaner
        self.config = config

    def run(self):
        keys = self.config['lists']
        self.logger.info('{}: GATHERING STATS ON EVENTS COUNT DIFFERENCES.'
                         .format(self.__class__.__name__))
        self.logger.info('All candidates lists: {}.'.format(keys))
        self.logger.info('Total candidates lists: {}.'.format(len(keys)))
        urls = {
            key: self.config['lists_json_url'].format(key)
            for key in keys
        }
        self._run_internal(urls=urls, stats_path=self.config['snapshot_path'])
        self.pusher.commit()
        self.logger.info('STATS ON EVENT COUNT DIFFERENCES ARE SENT.')

    def _collect(self, new_stats, old_stats):
        data = self.cleaner.clean(self._collect_internal(new_stats, old_stats))
        if self.logger.isEnabledFor(logging.DEBUG):
            with pandas.option_context('display.max_rows', len(data) + 1, 'display.width', 400):
                self.logger.debug('Collected differences:\n{}'.format(data))
        return data

    def _collect_internal(self, new_stats, old_stats):
        data = []
        for key in set(new_stats) & set(old_stats):
            self.logger.info('Processing key: {}'.format(key))
            old_bannerids_stats = old_stats[key]['statistic']['bannerids']
            new_bannerids_stats = new_stats[key]['statistic']['bannerids']
            old_key_ts = old_stats[key]['update_ts']
            new_key_ts = new_stats[key]['update_ts']
            for bannerid in set(new_bannerids_stats) & set(old_bannerids_stats):
                self.logger.debug('Processing bannerid {}'.format(bannerid))
                old_candidate = old_bannerids_stats[bannerid]
                new_candidate = new_bannerids_stats[bannerid]
                old_banner_update_ts = old_candidate['banner_update_ts']
                new_banner_update_ts = new_candidate['banner_update_ts']
                if new_banner_update_ts > old_banner_update_ts and new_banner_update_ts > old_key_ts:
                    old_host = old_candidate['host']
                    new_host = new_candidate['host']
                    old_product = old_candidate['product']
                    new_product = new_candidate['product']
                    if old_host == new_host and old_product == new_product:
                        old_bins = old_candidate['bins']
                        new_bins = new_candidate['bins']
                        for bin in set(new_bins) & set(old_bins):
                            old_counts = old_bins[bin]
                            new_counts = new_bins[bin]
                            for event in set(new_counts) & set(old_counts):
                                derivative, _ = self._compute_derivative(
                                    new_counts[event],
                                    old_counts[event],
                                    new_banner_update_ts,
                                    old_banner_update_ts
                                )
                                region, referer, subclient, client, browser, os = bin.split(',')
                                data.append({
                                    'list': key,
                                    'host': new_host,
                                    'product': new_product,
                                    'bannerid': bannerid,
                                    'event': event,
                                    'region': region,
                                    'referer': referer,
                                    'client': client,
                                    'subclient': subclient,
                                    'browser': browser,
                                    'os': os,
                                    'count': derivative,
                                    'update_ts': new_key_ts,
                                    'banner_update_ts': new_banner_update_ts
                                })
        return pandas.DataFrame(
            data,
            columns=['list', 'host', 'product', 'bannerid', 'event',
                     'region', 'referer', 'client', 'subclient', 'browser', 'os',
                     'count', 'update_ts', 'banner_update_ts']
        )
