# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

from .atom_monitorer_base import AtomMonitorerBase


class AtomDiffMonitorerBase(AtomMonitorerBase):
    def __init__(self, pusher, provider, holder, merger, aggregator):
        super(AtomDiffMonitorerBase, self).__init__(pusher, provider)
        self.logger = logging.getLogger(__name__)
        self.holder = holder
        self.merger = merger
        self.aggregator = aggregator

    def _run_internal(self, urls, stats_path):
        old_stats = self._load_stats(stats_path)
        new_stats = self._update_stats(old_stats, urls)
        differences = self._compute_differences(new_stats, old_stats)
        self.pusher.push(differences)
        self._save_stats(stats_path, new_stats)

    def _load_stats(self, path):
        return self.holder.load(path)

    def _save_stats(self, path, stats):
        self.holder.save(path, stats)

    def _update_stats(self, old_stats, urls):
        new_stats = self._get_new_stats(urls)
        return self.merger.merge(old_stats, new_stats)

    def _compute_differences(self, new_stats, old_stats):
        return self.aggregator.aggregate(self._collect(new_stats, old_stats))

    def _collect(self, new_stats, old_stats):
        raise NotImplementedError('{}._collect'.format(self.__class__.__name__))

    def _compute_derivative(self, new_value, old_value, new_ts, old_ts, clip_negative=True):
        delta_t = float(new_ts - old_ts) / 60.
        delta_val = new_value - old_value
        if clip_negative:
            delta_val = max(delta_val, 0.)
        if new_ts == old_ts:
            self.logger.warning('Wrong arguments old_ts={} is equal to new_ts={}.'.format(old_ts, new_ts))
            return 0., 0.
        if new_ts < old_ts:
            self.logger.warning('Wrong arguments old_ts={} is greater than new_ts={}.'.format(old_ts, new_ts))
        return delta_val / delta_t, delta_t
