# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

from nile.api.v1 import clusters, files
from .nile_monitorer import NileMonitorer
from ..utils.time import parse_date, shift_date


class AtomDistributionAlertsMonitorer(NileMonitorer):
    def __init__(self, date, leap, pusher, provider, cleaner, config):
        super(AtomDistributionAlertsMonitorer, self).__init__(pusher, config)
        self.logger = logging.getLogger(__name__)
        self.date = date
        self.leap = leap
        self.provider = provider
        self.cleaner = cleaner

    def run(self):
        self.logger.info('{}: GATHERING ALERTS ON SLICES'.format(self.__class__.__name__))
        self.logger.info('DATE: {}'.format(self.date))
        self.logger.info('LEAP: {}'.format(self.leap))
        self._run_internal()
        self.logger.info('ALERTS ON SLICES ARE SENT.')

    def _construct_job(self, cluster, job):
        pattern = self.config['sources']['totals']['path_pattern']
        source_then = job.table(self._get_path(pattern, self.date, -self.leap))
        source_now = job.table(self._get_path(pattern, self.date))
        for user_name, user_config in self.config['users'].items():
            tasks = user_config['tasks']
            self.logger.info('Performing tasks for user {}'.format(user_name))
            self.logger.info('There are {} tasks'.format(len(tasks)))
            for task_config in tasks:
                task_name = task_config['task_name']
                self.logger.info('Performing task {}'.format(task_name))
                then = self.provider.get(source_then, task_config['provider'])
                now = self.provider.get(source_now, task_config['provider'])
                alerts = self.cleaner.clean(then, now, task_config['cleaner'])
                alerts.put(self.__get_alerts_path(user_name, task_name))
            self.logger.info('All tasks completed')

    def _post_process(self, cluster):
        for user_name, user_config in self.config['users'].items():
            user_alerts = []
            for task_config in user_config['tasks']:
                task_name = task_config['task_name']
                alerts_path = self.__get_alerts_path(user_name, task_name)
                self.logger.info('Reading alerts from {} for user: {}, task: {}'.format(
                    alerts_path, user_name, task_name
                ))
                user_alerts.extend(cluster.read(alerts_path))
                cluster.driver.remove(alerts_path)
            self.pusher.push(user_alerts, user_config['pusher'])

    def __get_alerts_path(self, user_name, task_name):
        return '{}/{}_task_{}'.format(self.config['root'], user_name, task_name)

