# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

from .monitorer_base import MonitorerBase


class ClickhouseAtomMonitorer(MonitorerBase):
    def __init__(self, action, pusher, config):
        super(ClickhouseAtomMonitorer, self).__init__(pusher)
        self.logger = logging.getLogger(__name__)
        self.action = action
        self.config = config

    def run(self):
        self.logger.info(
            '{}: MODIFYING {} DATABASE. MODE: {}'.format(
                self.__class__.__name__,
                self.config['db_name'],
                self.action
            )
        )
        self._run_internal(self.action)
        self.logger.info('ACTIONS PERFORMED')

    def _run_internal(self, action):
        actions = {
            'build': self.__build_db,
            'rebuild': self.__rebuild_db,
            'create': self.__create_db,
            'drop': self.__drop_db,
            'optimize': self.__optimize_db,
            'init': self.__init_db
        }
        if action not in actions:
            self.logger.error('Action {} is not supported'.format(action))
            return
        actions.get(action)()

    def __build_db(self):
        self.__create_db()
        self.__init_db()

    def __rebuild_db(self):
        self.__drop_db()
        self.__build_db()

    def __create_db(self):
        self.logger.info('Creating database {}'.format(self.config['db_name']))
        self.__make_query('CREATE DATABASE {}'.format(self.config['db_name']))

    def __drop_db(self):
        self.logger.info('Dropping database {}'.format(self.config['db_name']))
        self.__make_query('DROP DATABASE {}'.format(self.config['db_name']))

    def __init_db(self):
        self.logger.info('Initializing database {}'.format(self.config['db_name']))
        for name in [self.config['abs_name'], self.config['diffs_name']]:
            self.__init_table(
                self.config['db_name'], name, self.__get_default_table_signature()
            )

    def __init_table(self, db_name, table_name, signature):
        self.logger.info('Building table {}.{}'.format(db_name, table_name))
        self.__make_query('CREATE TABLE {}.{} {}'.format(db_name, table_name, signature))

    def __get_default_table_signature(self):
        return (
            '( id Int32,  event_date Date,  host String,  product String, '
            'event String,  region String,  referer String,  client String, '
            'subclient String,  browser String,  os String,  lists Array(String), '
            'update_ts DateTime,  count Float32) '
            'ENGINE = MergeTree(event_date, (event_date, id, update_ts), 8192)'
        )

    def __optimize_db(self):
        self.logger.info('Optimizing database')
        for name in [self.config['abs_name'], self.config['diffs_name']]:
            self.__optimize_table(self.config['db_name'], name)

    def __optimize_table(self, db_name, table_name):
        self.logger.info('Building table {}.{}'.format(db_name, table_name))
        self.__make_query('OPTIMIZE TABLE {}.{}'.format(db_name, table_name))

    def __make_query(self, query):
        return self.pusher.push('POST', query)
