# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

from .atom_monitorer_base import MonitorerBase
from .atom_diff_monitorer import AtomDiffMonitorer
from ..pushers.integral_distribution_metric_slices_memory_pusher import IntegralDistributionMetricSlicesMemoryPusher
from ..cleaners.detailed_table_cleaner import DetailedTableCleaner
from ..mergers.rtmr_detailed_json_merger import RtmrDetailedJsonMerger
from ..aggregators.atom_slices_aggregator import AtomSlicesAggregator


class IntegralDistributionMetricMonitorer(MonitorerBase):
    def __init__(self, pusher, provider, holder, weighter, aggregator, config):
        super(IntegralDistributionMetricMonitorer, self).__init__(pusher)
        self.logger = logging.getLogger(__name__)
        self.provider = provider
        self.holder = holder
        self.weighter = weighter
        self.aggregator = aggregator
        self.config = config

    def run(self):
        self.logger.info(
            '{}: GATHERING STATS FOR INTEGRAL DISTRIBUTION METRIC'.format(
                self.__class__.__name__
            )
        )
        report = self.aggregator.aggregate({
            'weights': self.weighter.get(),
            'stats': self.__collect_stats(self.config)
        })
        self.pusher.push(report)
        self.pusher.commit()

    def __collect_stats(self, config):
        self.logger.info('Collecting stats')
        return {
            'installs': self.__collect_installs_stats(self.config['installs_config']),
            'shows': self.__collect_shows_stats(self.config['shows_config']),
            'closes': self.__collect_closes_stats(self.config['closes_config'])
        }

    def __collect_installs_stats(self, config):
        self.logger.info('Collecting stats on installs')
        return self.__collect_events_stats(config)

    def __collect_shows_stats(self, config):
        self.logger.info('Collecting stats on shows')
        return self.__collect_events_stats(config)

    def __collect_closes_stats(self, config):
        self.logger.info('Collecting stats on closes')
        return self.__collect_events_stats(config)

    def __collect_events_stats(self, config):
        pusher = IntegralDistributionMetricSlicesMemoryPusher()
        monitorer = AtomDiffMonitorer(
            pusher=pusher,
            provider=self.provider,
            holder=self.holder,
            merger=RtmrDetailedJsonMerger(),
            cleaner=DetailedTableCleaner(config['cleaner']),
            aggregator=AtomSlicesAggregator(config['aggregator']),
            config=config['monitorer']
        )
        monitorer.run()
        return pusher.get_value()
