# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

from .rtmr_json_parser_base import RtmrJsonParserBase
from .showid_parser import ShowidParser
from ..utils.time import get_now_ts


class RtmrDetailedJsonParser(RtmrJsonParserBase):
    def __init__(self, config):
        super(RtmrDetailedJsonParser, self).__init__()
        self.config = config
        self.logger = logging.getLogger(__name__)
        self.showid_parser = ShowidParser(self.config['contexts'])

    def parse(self, text):
        candidates_json = super(RtmrDetailedJsonParser, self).parse(text)
        try:
            parsed = self.__extract_data(candidates_json)
            bannerids = {}
            for name, stat in parsed['statistic'].items():
                if '_SUBST' not in name:
                    history = self.__parse_candidates_stat(stat)
                    history['host'] = self.__class__._get_host_by_candidate(name)
                    bannerid = self.__class__._get_bannerid_by_candidate(name)
                    bannerids[bannerid] = history
            parsed['statistic'] = {'bannerids': bannerids}
        except Exception as error:
            self.logger.exception('Failed to parse json. Error: {}'.format(error))
            parsed = {
                'statistic': {'bannerids': {}},
                'update_ts': get_now_ts()
            }
        return parsed

    def __extract_data(self, candidates_json):
        entry = candidates_json['Entries'][0]
        return {
            'statistic': super(RtmrDetailedJsonParser, self).parse(entry['Value']),
            'update_ts': self.__class__._get_ts(int(entry['SubKey']))
        }

    def __parse_candidates_stat(self, stat):
        details = super(RtmrDetailedJsonParser, self).parse(stat['v'])
        history = self._parse_counts(details['total'], extract_table={'__tag': 'product'})
        history['bins'] = {
            self.__parse_bin(bin): self._parse_counts(counts)['counts']
            for bin, counts in details['bins'].items()
        }
        history['banner_update_ts'] = self.__class__._get_ts(stat['ts'])
        return history

    def __parse_bin(self, bin):
        region, referer, showid = bin.split(',')
        region = region or 'unknown'
        referer = referer or 'unknown'
        showid = showid if showid and showid.isdigit() else '000000'
        subclient, client, browser, os = self.showid_parser.parse(showid, default='unknown')
        # subclient = self.config['contexts']['subclient'].get(showid[:2], 'unknown')
        # client = self.config['contexts']['client'].get(showid[2:4], 'unknown')
        # browser = self.config['contexts']['browser'].get(showid[4:6], 'unknown')
        # os = self.config['contexts']['os'].get(showid[6:8], 'unknown')
        return ','.join([region, referer, subclient, client, browser, os])
